//  PCAN-UDS.pas
//
//  ~~~~~~~~~~~~
//
//  PCAN-UDS API
//
//  ~~~~~~~~~~~~
//
//  ------------------------------------------------------------------
//  Author : Fabrice Vergnaud
//	Last changed by:	$Author: Fabrice $
//  Last changed date:	$Date: 2013-11-22 14:02:18 +0100 (ven., 22 nov. 2013) $
//
//  Language: Delphi
//  ------------------------------------------------------------------
//
//  Copyright (C) 2015  PEAK-System Technik GmbH, Darmstadt
//  more Info at http://www.peak-system.com
//

unit PUDS_2006;

interface

// Aliases definition
//
type
    TPUDSCANHandle = Word;     // Represents a CAN hardware channel of the underlying CAN system

{$Z2}
    /// <summary>
    /// Represents a PCAN Baud rate register value
    /// </summary>
    TPUDSBaudrate = (
        /// <summary>
        /// 1 MBit/s
        /// </summary>
        PUDS_BAUD_1M = $0014,
        /// <summary>
        /// 800 kBit/s
        /// </summary>
        PUDS_BAUD_800K = $0016,
        /// <summary>
        /// 500 kBit/s
        /// </summary>
        PUDS_BAUD_500K = $001C,
        /// <summary>
        /// 250 kBit/s
        /// </summary>
        PUDS_BAUD_250K = $011C,
        /// <summary>
        /// 125 kBit/s
        /// </summary>
        PUDS_BAUD_125K = $031C,
        /// <summary>
        /// 100 kBit/s
        /// </summary>
        PUDS_BAUD_100K = $432F,
        /// <summary>
        /// 95,238 kBit/s
        /// </summary>
        PUDS_BAUD_95K = $C34E,
        /// <summary>
        /// 83,333 kBit/s
        /// </summary>
        PUDS_BAUD_83K = $852B,
        /// <summary>
        /// 50 kBit/s
        /// </summary>
        PUDS_BAUD_50K = $472F,
        /// <summary>
        /// 47,619 kBit/s
        /// </summary>
        PUDS_BAUD_47K = $1414,
        /// <summary>
        /// 33,333 kBit/s
        /// </summary>
        PUDS_BAUD_33K = $8B2F,
        /// <summary>
        /// 20 kBit/s
        /// </summary>
        PUDS_BAUD_20K = $532F,
        /// <summary>
        /// 10 kBit/s
        /// </summary>
        PUDS_BAUD_10K = $672F,
        /// <summary>
        /// 5 kBit/s
        /// </summary>
        PUDS_BAUD_5K = $7F7F
    );

{$Z1}
    /// <summary>
    /// Represents the different Not Plug-And-Play PCAN Hardware types
    /// </summary>
    TPUDSHWType = (
        /// <summary>
        /// PCAN-ISA 82C200
        /// </summary>
        PUDS_TYPE_ISA = $01,
        /// <summary>
        /// PCAN-ISA SJA1000
        /// </summary>
        PUDS_TYPE_ISA_SJA = $09,
        /// <summary>
        /// PHYTEC ISA
        /// </summary>
        PUDS_TYPE_ISA_PHYTEC = $04,
        /// <summary>
        /// PCAN-Dongle 82C200
        /// </summary>
        PUDS_TYPE_DNG = $02,
        /// <summary>
        /// PCAN-Dongle EPP 82C200
        /// </summary>
        PUDS_TYPE_DNG_EPP = $03,
        /// <summary>
        /// PCAN-Dongle SJA1000
        /// </summary>
        PUDS_TYPE_DNG_SJA = $05,
        /// <summary>
        /// PCAN-Dongle EPP SJA1000
        /// </summary>
        PUDS_TYPE_DNG_SJA_EPP = $06
    );

{$Z4}
    /// <summary>
    /// Represent the PUDS error and status codes
    /// </summary>
    TPUDSStatus = (
        /// <summary>
        /// No error
        /// </summary>
        PUDS_ERROR_OK = $00000,
        /// <summary>
        /// Not Initialized
        /// </summary>
        PUDS_ERROR_NOT_INITIALIZED = $00001,
        /// <summary>
        /// Already Initialized
        /// </summary>
        PUDS_ERROR_ALREADY_INITIALIZED = $00002,
        /// <summary>
        /// Could not obtain memory
        /// </summary>
        PUDS_ERROR_NO_MEMORY = $00003,
        /// <summary>
        /// Input buffer overflow
        /// </summary>
        PUDS_ERROR_OVERFLOW = $00004,
        /// <summary>
        /// Timeout while accessing the PCANTP mutex
        /// </summary>
        PUDS_ERROR_TIMEOUT = $00006,
        /// <summary>
        /// No Message available
        /// </summary>
        PUDS_ERROR_NO_MESSAGE = $00007,
        /// <summary>
        /// Wrong message parameters
        /// </summary>
        PUDS_ERROR_WRONG_PARAM = $00008,
        /// <summary>
        /// PCANTP Channel is in BUS-LIGHT error state
        /// </summary>
        PUDS_ERROR_BUSLIGHT = $00009,
        /// <summary>
        /// PCANTP Channel is in BUS-HEAVY error state
        /// </summary>
        PUDS_ERROR_BUSHEAVY = $0000A,
        /// <summary>
        /// PCANTP Channel is in BUS-OFF error state
        /// </summary>
        PUDS_ERROR_BUSOFF = $0000B,
        /// <summary>
        /// Global CAN error, status code for composition of PCANBasic Errors.
        ///	Remove this value to get a PCAN-Basic TPCANStatus error code.
        /// </summary>
        PUDS_ERROR_CAN_ERROR = Longint($80000000)
    );

{$Z1}
    /// <summary>
    /// Represents network result values as defined in ISO 15765-2
    /// </summary>
    TPUDSResult = (
        /// <summary>
        /// general error
        /// </summary>
        PUDS_RESULT_N_OK = $00,
        /// <summary>
        /// timeout occured between 2 frames transmission (sender and receiver side)
        /// </summary>
        PUDS_RESULT_N_TIMEOUT_A = $01,
        /// <summary>
        /// sender side timeout while waiting for flow control frame
        /// </summary>
        PUDS_RESULT_N_TIMEOUT_BS = $02,
        /// <summary>
        /// receiver side timeout while waiting for consecutive frame
        /// </summary>
        PUDS_RESULT_N_TIMEOUT_CR = $03,
        /// <summary>
        /// unexpected sequence number
        /// </summary>
        PUDS_RESULT_N_WRONG_SN = $04,
        /// <summary>
        /// invalid or unknown FlowStatus
        /// </summary>
        PUDS_RESULT_N_INVALID_FS = $05,
        /// <summary>
        /// unexpected protocol data unit
        /// </summary>
        PUDS_RESULT_N_UNEXP_PDU = $06,
        /// <summary>
        /// reception of flow control WAIT frame that exceeds the maximum counter defined by PUDS_PARAM_WFT_MAX
        /// </summary>
        PUDS_RESULT_N_WFT_OVRN = $07,
        /// <summary>
        /// buffer on the receiver side cannot store the data length (server side only)
        /// </summary>
        PUDS_RESULT_N_BUFFER_OVFLW = $08,
        /// <summary>
        /// general error
        /// </summary>
        PUDS_RESULT_N_ERROR = $09
    );

    /// <summary>
    /// PCANTP parameters
    /// </summary>
    TPUDSParameter = (
        /// <summary>
        /// 2 Byte data describing the physical address of the equipment
        /// </summary>
        PUDS_PARAM_SERVER_ADDRESS = $C1,
        /// <summary>
        /// 2 Byte data (2 Byte functional address and MSB for status)
        /// describing a functional address to ignore or listen to
        /// </summary>
        PUDS_PARAM_SERVER_FILTER = $C2,
        /// <summary>
        /// 4 Byte data describing the maximum time allowed by the client to transmit a request
        /// See ISO-15765-3 §6.3.2 : /\P2Can_Req
        /// </summary>
        PUDS_PARAM_TIMEOUT_REQUEST = $C3,
        /// <summary>
        /// 4 Byte data describing the maximum time allowed by the client to receive a response
        /// See ISO-15765-3 §6.3.2 : /\P2Can_Rsp
        /// </summary>
        PUDS_PARAM_TIMEOUT_RESPONSE = $C4,
        /// <summary>
        /// Require a pointer to a TPUDSSessionInfo structure
        /// </summary>
        PUDS_PARAM_SESSION_INFO = $C5,
        /// <summary>
        /// API version parameter
        /// </summary>
        PUDS_PARAM_API_VERSION = $C6,
        /// <summary>
        /// Define UDS receive-event handler, require a pointer to an event HANDLE.
        /// </summary>
        PUDS_PARAM_RECEIVE_EVENT = $C7,
        /// <summary>
        /// Define a new ISO-TP mapping, requires a pointer to TPUDSMsg containing
		/// the mapped CAN ID and CAN ID response in the DATA.RAW field.
        /// </summary>
        PUDS_PARAM_MAPPING_ADD = $C8,
        /// <summary>
        /// Remove an ISO-TP mapping, requires a pointer to TPUDSMsg containing the mapped CAN ID to remove.
        /// </summary>
        PUDS_PARAM_MAPPING_REMOVE = $C9,

        /// <summary>
        /// 1 Byte data describing the block size parameter (BS)
        /// </summary>
        PUDS_PARAM_BLOCK_SIZE = $E1,
        /// <summary>
        /// 1 Byte data describing the seperation time parameter (STmin)
        /// </summary>
        PUDS_PARAM_SEPERATION_TIME = $E2,
        /// <summary>
        /// 1 Byte data describing the debug mode
        /// </summary>
        PUDS_PARAM_DEBUG = $E3,
        /// <summary>
        /// 1 Byte data describing the condition of a channel
        /// </summary>
        PUDS_PARAM_CHANNEL_CONDITION = $E4,
        /// <summary>
        /// Integer data describing the Wait Frame Transmissions parameter.
        /// </summary>
        PUDS_PARAM_WFT_MAX = $E5,
        /// <summary>
        /// 1 BYTE data stating if CAN frame DLC uses padding or not
        /// </summary>
        PUDS_PARAM_CAN_DATA_PADDING = $E8,
        /// <summary>
        /// 1 BYTE data stating the value used for CAN data padding
        /// </summary>
        PUDS_PARAM_PADDING_VALUE = $ED
    );

    /// <summary>
    /// PUDS Service IDs defined in ISO 14229-1
    /// </summary>
    TPUDSService = (
        PUDS_SI_DiagnosticSessionControl = $10,
        PUDS_SI_ECUReset = $11,
        PUDS_SI_SecurityAccess = $27,
        PUDS_SI_CommunicationControl = $28,
        PUDS_SI_TesterPresent = $3E,
        PUDS_SI_AccessTimingParameter = $83,
        PUDS_SI_SecuredDataTransmission = $84,
        PUDS_SI_ControlDTCSetting = $85,
        PUDS_SI_ResponseOnEvent = $86,
        PUDS_SI_LinkControl = $87,
        PUDS_SI_ReadDataByIdentifier = $22,
        PUDS_SI_ReadMemoryByAddress = $23,
        PUDS_SI_ReadScalingDataByIdentifier = $24,
        PUDS_SI_ReadDataByPeriodicIdentifier = $2A,
        PUDS_SI_DynamicallyDefineDataIdentifier = $2C,
        PUDS_SI_WriteDataByIdentifier = $2E,
        PUDS_SI_WriteMemoryByAddress = $3D,
        PUDS_SI_ClearDiagnosticInformation = $14,
        PUDS_SI_ReadDTCInformation = $19,
        PUDS_SI_InputOutputControlByIdentifier = $2F,
        PUDS_SI_RoutineControl = $31,
        PUDS_SI_RequestDownload = $34,
        PUDS_SI_RequestUpload = $35,
        PUDS_SI_TransferData = $36,
        PUDS_SI_RequestTransferExit = $37,
        /// <summary>
        /// Negative response code
        /// </summary>
        PUDS_NR_SI = $7f
    );

    /// <summary>
    /// PUDS ISO_15765_4 address definitions
    /// </summary>
    TPUDSAddress = (
        /// <summary>
        /// External test equipment
        /// </summary>
        PUDS_ISO_15765_4_ADDR_TEST_EQUIPMENT = $F1,
        /// <summary>
        /// OBD funtional system
        /// </summary>
        PUDS_ISO_15765_4_ADDR_OBD_FUNCTIONAL = $33,
        /// <summary>
        /// ECU 1
        /// </summary>
        PUDS_ISO_15765_4_ADDR_ECU_1 = $01,
        /// <summary>
        /// ECU 2
        /// </summary>
        PUDS_ISO_15765_4_ADDR_ECU_2 = $02,
        /// <summary>
        /// ECU 3
        /// </summary>
        PUDS_ISO_15765_4_ADDR_ECU_3 = $03,
        /// <summary>
        /// ECU 4
        /// </summary>
        PUDS_ISO_15765_4_ADDR_ECU_4 = $04,
        /// <summary>
        /// ECU 5
        /// </summary>
        PUDS_ISO_15765_4_ADDR_ECU_5 = $05,
        /// <summary>
        /// ECU 6
        /// </summary>
        PUDS_ISO_15765_4_ADDR_ECU_6 = $06,
        /// <summary>
        /// ECU 7
        /// </summary>
        PUDS_ISO_15765_4_ADDR_ECU_7 = $07,
        /// <summary>
        /// ECU 8
        /// </summary>
        PUDS_ISO_15765_4_ADDR_ECU_8 = $08
    );

{$Z4}
    /// <summary>
    /// PUDS ISO_15765_4 11 bit CAN Identifier
    /// </summary>
    TPUDSCanId = (
        /// <summary>
        /// CAN ID for functionally addressed request messages sent by external test equipment
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_FUNCTIONAL_REQUEST = $7DF,
        /// <summary>
        /// physical request CAN ID from external test equipment to ECU #1
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_1 = $7E0,
        /// <summary>
        /// physical response CAN ID from ECU #1 to external test equipment
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_1 = $7E8,
        /// <summary>
        /// physical request CAN ID from external test equipment to ECU #2
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_2 = $7E1,
        /// <summary>
        /// physical response CAN ID from ECU #2 to external test equipment
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_2 = $7E9,
        /// <summary>
        /// physical request CAN ID from external test equipment to ECU #3
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_3 = $7E2,
        /// <summary>
        /// physical response CAN ID from ECU #3 to external test equipment
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_3 = $7EA,
        /// <summary>
        /// physical request CAN ID from external test equipment to ECU #4
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_4 = $7E3,
        /// <summary>
        /// physical response CAN ID from ECU #4 to external test equipment
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_4 = $7EB,
        /// <summary>
        /// physical request CAN ID from external test equipment to ECU #5
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_5 = $7E4,
        /// <summary>
        /// physical response CAN ID from ECU #5 to external test equipment
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_5 = $7EC,
        /// <summary>
        /// physical request CAN ID from external test equipment to ECU #6
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_6 = $7E5,
        /// <summary>
        /// physical response CAN ID from ECU #6 to external test equipment
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_6 = $7ED,
        /// <summary>
        /// physical request CAN ID from external test equipment to ECU #7
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_7 = $7E6,
        /// <summary>
        /// physical response CAN ID from ECU #7 to external test equipment
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_7 = $7EE,
        /// <summary>
        /// physical request CAN ID from external test equipment to ECU #8
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_REQUEST_8 = $7E7,
        /// <summary>
        /// physical response CAN ID from ECU #8 to external test equipment
        /// </summary>
        PUDS_ISO_15765_4_CAN_ID_PHYSICAL_RESPONSE_8 = $7EF
    );

{$Z1}
    /// <summary>
    /// PUDS Protocol ISO-15765 definitions
    /// </summary>
    TPUDSProtocol = (
        /// <summary>
        /// non ISO-TP frame (Unacknowledge Unsegmented Data Transfer)
        /// </summary>
	    PUDS_PROTOCOL_NONE = $00,
        /// <summary>
        /// using PCAN-ISO-TP with 11 BIT CAN ID, NORMAL addressing and diagnostic message
        /// </summary>
        PUDS_PROTOCOL_ISO_15765_2_11B = $01,
        /// <summary>
        /// using PCAN-ISO-TP with 11 BIT CAN ID, MIXED addressing and remote diagnostic message
        /// </summary>
        PUDS_PROTOCOL_ISO_15765_2_11B_REMOTE = $02,
        /// <summary>
        /// using PCAN-ISO-TP with 29 BIT CAN ID, FIXED NORMAL addressing and diagnostic message
        /// </summary>
        PUDS_PROTOCOL_ISO_15765_2_29B = $03,
        /// <summary>
        /// using PCAN-ISO-TP with 29 BIT CAN ID, MIXED addressing and remote diagnostic message
        /// </summary>
        PUDS_PROTOCOL_ISO_15765_2_29B_REMOTE = $04,
        /// <summary>
        /// using PCAN-ISO-TP with Enhanced diagnostics 29 bit CAN Identifiers
        /// </summary>
        PUDS_PROTOCOL_ISO_15765_3_29B = $05,
        /// <summary>
        /// using PCAN-ISO-TP with 29 BIT CAN ID, NORMAL addressing and diagnostic message
		/// Note: this protocol requires extra mapping definitions via PCAN-ISO-TP API
        /// </summary>
        PUDS_PROTOCOL_ISO_15765_2_29B_NORMAL = $06,
        /// <summary>
        /// using PCAN-ISO-TP with 11 BIT CAN ID, EXTENDED addressing and diagnostic message
		/// Note: this protocol requires extra mapping definitions via PCAN-ISO-TP API
        /// </summary>
        PUDS_PROTOCOL_ISO_15765_2_11B_EXTENDED = $07,
        /// <summary>
        /// using PCAN-ISO-TP with 29 BIT CAN ID, EXTENDED addressing and diagnostic message
		/// Note: this protocol requires extra mapping definitions via PCAN-ISO-TP API
        /// </summary>
        PUDS_PROTOCOL_ISO_15765_2_29B_EXTENDED = $08
    );

    /// <summary>
    /// PUDS addressing type
    /// </summary>
    TPUDSAddressingType = (
        /// <summary>
        /// Physical addressing
        /// </summary>
        PUDS_ADDRESSING_PHYSICAL = $01,
        /// <summary>
        /// Functional addressing
        /// </summary>
        PUDS_ADDRESSING_FUNCTIONAL = $02
    );

    /// <summary>
    /// PCANTP message types
    /// </summary>
    TPUDSMessageType = (
        /// <summary>
        /// UDS Request Message
        /// </summary>
        PUDS_MESSAGE_TYPE_REQUEST = $00,
        /// <summary>
        /// UDS Request/Response confirmation  Message
        /// </summary>
        PUDS_MESSAGE_TYPE_CONFIRM = $01,
        /// <summary>
        /// Incoming UDS Message
        /// </summary>
        PUDS_MESSAGE_TYPE_INDICATION = $02,
        /// <summary>
        /// UDS Message transmission started
        /// </summary>
        PUDS_MESSAGE_TYPE_INDICATION_TX = $03,
        /// <summary>
        /// Unacknowledge Unsegmented Data Transfert
        /// </summary>
        PUDS_MESSAGE_TYPE_CONFIRM_UUDT = $04
    );

	/// <summary>
	/// PCAN-UDS Network Addressing Information
	/// </summary>
{$A8}
    TPUDSNetAddrInfo = record
        /// <summary>
        /// Represents the origin of this message (address from
        /// where this message was or will be sent)
        /// </summary>
        SA: Byte;
        /// <summary>
        /// Represents the destination of this message (address to
        /// where this message was or will be sent)
        /// </summary>
        TA: Byte;
        /// <summary>
        /// Represents the kind of addressing being used for communication
        /// </summary>
        TA_TYPE: TPUDSAddressingType;
        /// <summary>
        /// Represents the destination of this message in a remote network
        /// </summary>
        RA: Byte;
        /// <summary>
        /// Represents the protocol being used for communication
        /// </summary>
        PROTOCOL: TPUDSProtocol;
    end;

	/// <summary>
	/// PCAN-UDS Diagnostic Session Information of a server
	/// </summary>
{$A8}
    TPUDSSessionInfo = record
        /// <summary>
        /// Network address information
        /// </summary>
	    NETADDRINFO: TPUDSNetAddrInfo;
        /// <summary>
        /// Activated Diagnostic Session (see PUDS_SVC_PARAM_DSC_xxx values)
        /// </summary>
	    SESSION_TYPE: Byte;
        /// <summary>
        /// Default P2Can_Server_Max timing for the activated session
        /// </summary>
	    TIMEOUT_P2CAN_SERVER_MAX: Word;
        /// <summary>
        /// Enhanced P2Can_Server_Max timing for the activated session
        /// </summary>
	    TIMEOUT_ENHANCED_P2CAN_SERVER_MAX: Word;
    end;

	/// <summary>
	/// PCAN-UDS Message
	/// </summary>
{$A8}
    TPUDSMsg = record
        /// <summary>
        /// Network Addressing Information
        /// </summary>
	    NETADDRINFO: TPUDSNetAddrInfo;
        /// <summary>
        /// Result status of the network communication
        /// </summary>
	    RESULT: TPUDSResult;
        /// <summary>
        /// States wether Positive Response Message should be suppressed.
		/// See constants PUDS_SUPPR_POS_RSP_MSG_INDICATION_BIT & PUDS_KEEP_POS_RSP_MSG_INDICATION_BIT
        /// </summary>
	    NO_POSITIVE_RESPONSE_MSG: Byte;
        /// <summary>
        /// Data Length of the message
        /// </summary>
	    LEN: Word;
        /// <summary>
        /// Type of UDS Message
        /// </summary>
	    MSGTYPE: TPUDSMessageType;
        /// <summary>
        /// Represents the buffer containing the data of this message
        /// </summary>
        DATA: array[0..4095] of Byte;
    end;
    PTPUDSMsg = ^TPUDSMsg;



////////////////////////////////////////////////////////////////////////////////
///    UDS Service Parameters
///

{$Z1}
        /// <summary>
        /// Subfunction parameter for UDS service DiagnosticSessionControl
        /// </summary>
        TPUDSSvcParamDSC = (
            /// <summary>
            /// Default Session
            /// </summary>
            PUDS_SVC_PARAM_DSC_DS = $01,
            /// <summary>
            /// ECU Programming Session
            /// </summary>
            PUDS_SVC_PARAM_DSC_ECUPS = $02,
            /// <summary>
            /// ECU Extended Diagnostic Session
            /// </summary>
            PUDS_SVC_PARAM_DSC_ECUEDS = $03,
            /// <summary>
            /// Safety System Diagnostic Session
            /// </summary>
            PUDS_SVC_PARAM_DSC_SSDS = $04
        );

{$Z1}
        /// <summary>
        /// Subfunction parameter for UDS service ECURest
        /// </summary>
        TPUDSSvcParamER = (
            /// <summary>
            /// Hard Reset
            /// </summary>
            PUDS_SVC_PARAM_ER_HR = $01,
            /// <summary>
            /// Key Off on Reset
            /// </summary>
            PUDS_SVC_PARAM_ER_KOFFONR = $02,
            /// <summary>
            /// Soft Reset
            /// </summary>
            PUDS_SVC_PARAM_ER_SR = $03,
            /// <summary>
            /// Enable Rapid Power Shutdown
            /// </summary>
            PUDS_SVC_PARAM_ER_ERPSD = $04,
            /// <summary>
            /// Disable Rapid Power Shutdown
            /// </summary>
            PUDS_SVC_PARAM_ER_DRPSD = $05
        );

{$Z1}
        /// <summary>
        /// ControlType: Subfunction parameter for UDS service CommunicationControl
        /// </summary>
        TPUDSSvcParamCC = (
            /// <summary>
            /// Enable Rx and Tx
            /// </summary>
            PUDS_SVC_PARAM_CC_ERXTX = $00,
            /// <summary>
            /// Enable Rx and Disable Tx
            /// </summary>
            PUDS_SVC_PARAM_CC_ERXDTX = $01,
            /// <summary>
            /// Disable Rx and Enable Tx
            /// </summary>
            PUDS_SVC_PARAM_CC_DRXETX = $02,
            /// <summary>
            /// Disable Rx and Tx
            /// </summary>
            PUDS_SVC_PARAM_CC_DRXTX = $03
        );

{$Z1}
        /// <summary>
        /// TesterPresentType: Subfunction parameter for UDS service TesterPresent
        /// </summary>
        TPUDSSvcParamTP = (
            /// <summary>
            /// Zero SubFunction
            /// </summary>
            PUDS_SVC_PARAM_TP_ZSUBF = $00
        );

{$Z1}
        /// <summary>
        /// DTCSettingType: Subfunction parameter for UDS service ControlDTCSetting
        /// ISO
        /// </summary>
        TPUDSSvcParamCDTCS = (
            /// <summary>
            /// The server(s) shall resume the setting of diagnostic trouble codes
            /// </summary>
            PUDS_SVC_PARAM_CDTCS_ON = $01,
            /// <summary>
            /// The server(s) shall stop the setting of diagnostic trouble codes
            /// </summary>
            PUDS_SVC_PARAM_CDTCS_OFF = $02
        );

{$Z1}
        /// <summary>
        /// EventType: Subfunction parameter for UDS service ResponseOnEvent
        /// </summary>
        TPUDSSvcParamROE = (
            /// <summary>
            /// Stop Response On Event
            /// </summary>
            PUDS_SVC_PARAM_ROE_STPROE = $00,
            /// <summary>
            /// On DTC Status Change
            /// </summary>
            PUDS_SVC_PARAM_ROE_ONDTCS = $01,
            /// <summary>
            /// On Timer Interrupt
            /// </summary>
            PUDS_SVC_PARAM_ROE_OTI = $02,
            /// <summary>
            /// On Change Of Data Identifier
            /// </summary>
            PUDS_SVC_PARAM_ROE_OCODID = $03,
            /// <summary>
            /// Report Activated Events
            /// </summary>
            PUDS_SVC_PARAM_ROE_RAE = $04,
            /// <summary>
            /// Start Response On Event
            /// </summary>
            PUDS_SVC_PARAM_ROE_STRTROE = $05,
            /// <summary>
            /// Clear Response On Event
            /// </summary>
            PUDS_SVC_PARAM_ROE_CLRROE = $06,
            /// <summary>
            /// On Comparison Of Values
            /// </summary>
            PUDS_SVC_PARAM_ROE_OCOV = $07
        );
  {$Z1}
        /// <summary>
        /// RoE Recommended service (first Byte of ServiceToRespondTo Record)
        /// </summary>
        TPUDSSvcParamROERecommendedServiceID = (
            PUDS_SVC_PARAM_ROE_STRT_SI_RDBI = 34,   // TPUDSService.PUDS_SI_ReadDataByIdentifier,
            PUDS_SVC_PARAM_ROE_STRT_SI_RDTCI = 25,  // TPUDSService.PUDS_SI_ReadDTCInformation,
            PUDS_SVC_PARAM_ROE_STRT_SI_RC = 49,     // TPUDSService.PUDS_SI_RoutineControl,
            PUDS_SVC_PARAM_ROE_STRT_SI_IOCBI = 47  // TPUDSService.PUDS_SI_InputOutputControlByIdentifier
        );

{$Z1}
        /// <summary>
        /// LinkControlType: Subfunction parameter for UDS service LinkControl
        /// </summary>
        TPUDSSvcParamLC = (
            /// <summary>
            /// Verify Baudrate Transition With Fixed Baudrate
            /// </summary>
            PUDS_SVC_PARAM_LC_VBTWFBR = $01,
            /// <summary>
            /// Verify Baudrate Transition With Specific Baudrate
            /// </summary>
            PUDS_SVC_PARAM_LC_VBTWSBR = $02,
            /// <summary>
            /// Transition Baudrate
            /// </summary>
            PUDS_SVC_PARAM_LC_TB = $03
        );
{$Z1}
        /// <summary>
        /// BaudrateIdentifier: standard Baudrate Identifiers
        /// </summary>
        TPUDSSvcParamLCBaudrateIdentifier = (
            /// <summary>
            /// standard PC baud rate of 9.6 KBaud
            /// </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_PC_9600	=	$01,
            /// <summary>
            /// standard PC baud rate of 19.2 KBaud
            /// </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_PC_19200	=	$02,
            /// <summary>
            /// standard PC baud rate of 38.4 KBaud
            /// </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_PC_38400	=	$03,
            /// <summary>
            /// standard PC baud rate of 57.6 KBaud
            /// </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_PC_57600	=	$04,
            /// <summary>
            /// standard PC baud rate of 115.2 KBaud
            /// </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_PC_115200 =	$05,
            /// <summary>
            /// standard CAN baud rate of 125 KBaud
            /// </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_CAN_125K	=	$10,
            /// <summary>
            /// standard CAN baud rate of 250 KBaud
            /// </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_CAN_250K	=	$11,
            /// <summary>
            /// standard CAN baud rate of 500 KBaud
            /// </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_CAN_500K	=	$12,
            /// <summary>
            /// standard CAN baud rate of 1 MBaud
            /// </summary>
            PUDS_SVC_PARAM_LC_BAUDRATE_CAN_1M	=	$13
        );

{$Z2}
        /// <summary>
        /// Data Identifiers ISO-14229-1:2006 §C.1 p.259
        /// </summary>
        TPUDSSvcParamDI= (
            /// <summary>
            /// bootSoftwareIdentificationDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_BSIDID = $F180,
            /// <summary>
            /// applicationSoftwareIdentificationDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_ASIDID = $F181,
            /// <summary>
            /// applicationDataIdentificationDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_ADIDID = $F182,
            /// <summary>
            /// bootSoftwareIdentificationDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_BSFPDID = $F183,
            /// <summary>
            /// applicationSoftwareFingerprintDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_ASFPDID = $F184,
            /// <summary>
            /// applicationDataFingerprintDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_ADFPDID = $F185,
            /// <summary>
            /// activeDiagnosticSessionDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_ADSDID = $F186,
            /// <summary>
            /// vehicleManufacturerSparePartNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_VMSPNDID = $F187,
            /// <summary>
            /// vehicleManufacturerECUSoftwareNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_VMECUSNDID = $F188,
            /// <summary>
            /// vehicleManufacturerECUSoftwareVersionNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_VMECUSVNDID = $F189,
            /// <summary>
            /// systemSupplierIdentifierDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_SSIDDID = $F18A,
            /// <summary>
            /// ECUManufacturingDateDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_ECUMDDID = $F18B,
            /// <summary>
            /// ECUSerialNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_ECUSNDID = $F18C,
            /// <summary>
            /// supportedFunctionalUnitsDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_SFUDID = $F18D,
            /// <summary>
            /// vehicleManufacturerKitAssemblyPartNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_VMKAPNDID = $F18E,
            /// <summary>
            /// VINDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_VINDID = $F190,
            /// <summary>
            /// vehicleManufacturerECUHardwareNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_VMECUHNDID = $F191,
            /// <summary>
            /// systemSupplierECUHardwareNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_SSECUHWNDID = $F192,
            /// <summary>
            /// systemSupplierECUHardwareVersionNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_SSECUHWVNDID = $F193,
            /// <summary>
            /// systemSupplierECUSoftwareNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_SSECUSWNDID = $F194,
            /// <summary>
            /// systemSupplierECUSoftwareVersionNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_SSECUSWVNDID = $F195,
            /// <summary>
            /// exhaustRegulationOrTypeApprovalNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_EROTANDID = $F196,
            /// <summary>
            /// systemNameOrEngineTypeDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_SNOETDID = $F197,
            /// <summary>
            /// repairShopCodeOrTesterSerialNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_RSCOTSNDID = $F198,
            /// <summary>
            /// programmingDateDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_PDDID = $F199,
            /// <summary>
            /// calibrationRepairShopCodeOrCalibrationEquipmentSerialNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_CRSCOCESNDID = $F19A,
            /// <summary>
            /// calibrationDateDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_CDDID = $F19B,
            /// <summary>
            /// calibrationEquipmentSoftwareNumberDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_CESWNDID = $F19C,
            /// <summary>
            /// ECUInstallationDateDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_EIDDID = $F19D,
            /// <summary>
            /// ODXFileDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_ODXFDID = $F19E,
            /// <summary>
            /// entityDataIdentifier
            /// </summary>
            PUDS_SVC_PARAM_DI_EDID = $F19F
        );

{$Z1}
        /// <summary>
        /// TransmissionMode: Subfunction parameter for UDS service ReadDataByPeriodicIdentifier
        /// </summary>
        TPUDSSvcParamRDBPI = (
            /// <summary>
            /// Send At Slow Rate
            /// </summary>
            PUDS_SVC_PARAM_RDBPI_SASR = $01,
            /// <summary>
            /// Send At Medium Rate
            /// </summary>
            PUDS_SVC_PARAM_RDBPI_SAMR = $02,
            /// <summary>
            /// Send At Fast Rate
            /// </summary>
            PUDS_SVC_PARAM_RDBPI_SAFR = $03,
            /// <summary>
            /// Stop Sending
            /// </summary>
            PUDS_SVC_PARAM_RDBPI_SS = $04
        );

{$Z1}
        /// <summary>
        /// DynamicallyDefineDataIdentifier Type: Subfunction parameter for UDS service DynamicallyDefineDataIdentifier
        /// </summary>
        TPUDSSvcParamDDDI = (
            /// <summary>
            /// Define By Identifier
            /// </summary>
            PUDS_SVC_PARAM_DDDI_DBID = $01,
            /// <summary>
            /// Define By Memory Address
            /// </summary>
            PUDS_SVC_PARAM_DDDI_DBMA = $02,
            /// <summary>
            /// Clear Dynamically Defined Data Identifier
            /// </summary>
            PUDS_SVC_PARAM_DDDI_CDDDI = $03
        );

{$Z1}
        /// <summary>
        /// RDTCIType: Subfunction parameter for UDS service ReadDTCInformation
        /// ISO-15765-3:2004 §9.4.1 p.54 && ISO-14229-1:2006 §11.3 p.154
        /// </summary>
        TPUDSSvcParamRDTCI = (
            /// <summary>
            /// report Number Of DTC By Status Mask
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RNODTCBSM = $01,
            /// <summary>
            /// report DTC By Status Mask
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCBSM = $02,
            /// <summary>
            /// report DTC Snapshot Identification
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCSSI = $03,
            /// <summary>
            /// report DTC Snapshot Record By DTC Number
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCSSBDTC = $04,
            /// <summary>
            /// report DTC Snapshot Record By Record Number
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCSSBRN = $05,
            /// <summary>
            /// report DTC Extended Data Record By DTC Number
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN = $06,
            /// <summary>
            /// report Number Of DTC By Severity Mask Record
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RNODTCBSMR = $07,
            /// <summary>
            /// report DTC By Severity Mask Record
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCBSMR = $08,
            /// <summary>
            /// report Severity Information Of DTC
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RSIODTC = $09,
            /// <summary>
            /// report Supported DTC
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RSUPDTC = $0A,
            /// <summary>
            /// report First Test Failed DTC
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RFTFDTC = $0B,
            /// <summary>
            /// report First Confirmed DTC
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RFCDTC = $0C,
            /// <summary>
            /// report Most Recent Test Failed DTC
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RMRTFDTC = $0D,
            /// <summary>
            /// report Most Recent Confirmed DTC
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RMRCDTC = $0E,
            /// <summary>
            /// report Mirror Memory DTC By Status Mask
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RMMDTCBSM = $0F,
            /// <summary>
            /// report Mirror Memory DTC Extended Data Record By DTC Number
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RMMDEDRBDN = $10,
            /// <summary>
            /// report Number Of Mirror MemoryDTC By Status Mask
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RNOMMDTCBSM = $11,
            /// <summary>
            /// report Number Of Emissions Related OBD DTC By Status Mask
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RNOOBDDTCBSM = $12,
            /// <summary>
            /// report Emissions Related OBD DTC By Status Mask
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_ROBDDTCBSM = $13,
            /// <summary>
            /// report DTC Fault Detection Counter
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCFDC = $14,
            /// <summary>
            /// report DTC With Permanent Status
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_RDTCWPS = $15
        );
{$Z1}
        /// <summary>
        /// DTCSeverityMask (DTCSVM) : ISO-14229-1:2006 §D.3 p.285
        /// </summary>
        TPUDSSvcParamRDTCI_DTCSVM = (
            /// <summary>
            /// DTC severity bit definitions : no SeverityAvailable
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_DTCSVM_NSA = $00,
            /// <summary>
            /// DTC severity bit definitions : maintenance Only
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_DTCSVM_MO = $20,
            /// <summary>
            /// DTC severity bit definitions : check At Next Halt
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_DTCSVM_CHKANH = $40,
            /// <summary>
            /// DTC severity bit definitions : check Immediately
            /// </summary>
            PUDS_SVC_PARAM_RDTCI_DTCSVM_CHKI = $80
        );

{$Z1}
        /// <summary>
        /// inputOutputControlParameter: ISO-14229-1:2006  §E.1 p.289
        /// </summary>
        TPUDSSvcParamIOCBI = (
            /// <summary>
            /// returnControlToECU (0 controlState Bytes in request)
            /// </summary>
            PUDS_SVC_PARAM_IOCBI_RCTECU = $00,
            /// <summary>
            /// resetToDefault (0 controlState Bytes in request)
            /// </summary>
            PUDS_SVC_PARAM_IOCBI_RTD = $01,
            /// <summary>
            /// freezeCurrentState (0 controlState Bytes in request)
            /// </summary>
            PUDS_SVC_PARAM_IOCBI_FCS = $02,
            /// <summary>
            /// shortTermAdjustment
            /// </summary>
            PUDS_SVC_PARAM_IOCBI_STA = $03
        );

{$Z1}
        /// <summary>
        /// RoutineControlType: Subfunction parameter for UDS service RoutineControl
        /// </summary>
        TPUDSSvcParamRC = (
            /// <summary>
            /// Start Routine
            /// </summary>
            PUDS_SVC_PARAM_RC_STR = $01,
            /// <summary>
            /// Stop Routine
            /// </summary>
            PUDS_SVC_PARAM_RC_STPR = $02,
            /// <summary>
            /// Request Routine Results
            /// </summary>
            PUDS_SVC_PARAM_RC_RRR = $03
        );
{$Z2}
        /// <summary>
        /// Routine Identifier: ISO-14229-1:2006 §F.1 p.290
        /// </summary>
        TPUDSSvcParamRC_RID= (
            /// <summary>
            /// Deploy Loop Routine ID
            /// </summary>
            PUDS_SVC_PARAM_RC_RID_DLRI_ = $E200,
            /// <summary>
            /// erase Memory
            /// </summary>
            PUDS_SVC_PARAM_RC_RID_EM_ = $FF00,
            /// <summary>
            /// check Programming Dependencies
            /// </summary>
            PUDS_SVC_PARAM_RC_RID_CPD_ = $FF01,
            /// <summary>
            /// erase Mirror Memory DTCs
            /// </summary>
            PUDS_SVC_PARAM_RC_RID_EMMDTC_ = $FF02
        );



////////////////////////////////////////////////////////////////////////////////
///    UDS API Class
///

    TUdsApi = class
    public
        class var
        /// <summary>
        /// Undefined/default value for a PCAN bus
        /// </summary>
        const PUDS_NONEBUS: TPUDSCANHandle = $00;

        /// <summary>
        /// PCAN-ISA interface, channel 1
        /// </summary>
        const PUDS_ISABUS1: TPUDSCANHandle = $21;
        /// <summary>
        /// PCAN-ISA interface, channel 2
        /// </summary>
        const PUDS_ISABUS2: TPUDSCANHandle = $22;
        /// <summary>
        /// PCAN-ISA interface, channel 3
        /// </summary>
        const PUDS_ISABUS3: TPUDSCANHandle = $23;
        /// <summary>
        /// PCAN-ISA interface, channel 4
        /// </summary>
        const PUDS_ISABUS4: TPUDSCANHandle = $24;
        /// <summary>
        /// PCAN-ISA interface, channel 5
        /// </summary>
        const PUDS_ISABUS5: TPUDSCANHandle = $25;
        /// <summary>
        /// PCAN-ISA interface, channel 6
        /// </summary>
        const PUDS_ISABUS6: TPUDSCANHandle = $26;
        /// <summary>
        /// PCAN-ISA interface, channel 7
        /// </summary>
        const PUDS_ISABUS7: TPUDSCANHandle = $27;
        /// <summary>
        /// PCAN-ISA interface, channel 8
        /// </summary>
        const PUDS_ISABUS8: TPUDSCANHandle = $28;

        /// <summary>
        /// PPCAN-Dongle/LPT interface, channel 1
        /// </summary>
        const PUDS_DNGBUS1: TPUDSCANHandle = $31;

        /// <summary>
        /// PCAN-PCI interface, channel 1
        /// </summary>
        const PUDS_PCIBUS1: TPUDSCANHandle = $41;
        /// <summary>
        /// PCAN-PCI interface, channel 2
        /// </summary>
        const PUDS_PCIBUS2: TPUDSCANHandle = $42;
        /// <summary>
        /// PCAN-PCI interface, channel 3
        /// </summary>
        const PUDS_PCIBUS3: TPUDSCANHandle = $43;
        /// <summary>
        /// PCAN-PCI interface, channel 4
        /// </summary>
        const PUDS_PCIBUS4: TPUDSCANHandle = $44;
        /// <summary>
        /// PCAN-PCI interface, channel 5
        /// </summary>
        const PUDS_PCIBUS5: TPUDSCANHandle = $45;
        /// <summary>
        /// PCAN-PCI interface, channel 6
        /// </summary>
        const PUDS_PCIBUS6: TPUDSCANHandle = $46;
        /// <summary>
        /// PCAN-PCI interface, channel 7
        /// </summary>
        const PUDS_PCIBUS7: TPUDSCANHandle = $47;
        /// <summary>
        /// PCAN-PCI interface, channel 8
        /// </summary>
        const PUDS_PCIBUS8: TPUDSCANHandle = $48;
        /// <summary>
        /// PCAN-PCI interface, channel 9
        /// </summary>
        const PUDS_PCIBUS9: TPUDSCANHandle = $409;
        /// <summary>
        /// PCAN-PCI interface, channel 10
        /// </summary>
        const PUDS_PCIBUS10: TPUDSCANHandle = $40A;
        /// <summary>
        /// PCAN-PCI interface, channel 11
        /// </summary>
        const PUDS_PCIBUS11: TPUDSCANHandle = $40B;
        /// <summary>
        /// PCAN-PCI interface, channel 12
        /// </summary>
        const PUDS_PCIBUS12: TPUDSCANHandle = $40C;
        /// <summary>
        /// PCAN-PCI interface, channel 13
        /// </summary>
        const PUDS_PCIBUS13: TPUDSCANHandle = $40D;
        /// <summary>
        /// PCAN-PCI interface, channel 14
        /// </summary>
        const PUDS_PCIBUS14: TPUDSCANHandle = $40E;
        /// <summary>
        /// PCAN-PCI interface, channel 15
        /// </summary>
        const PUDS_PCIBUS15: TPUDSCANHandle = $40F;
        /// <summary>
        /// PCAN-PCI interface, channel 16
        /// </summary>
        const PUDS_PCIBUS16: TPUDSCANHandle = $410;

        /// <summary>
        /// PCAN-USB interface, channel 1
        /// </summary>
        const PUDS_USBBUS1: TPUDSCANHandle = $51;
        /// <summary>
        /// PCAN-USB interface, channel 2
        /// </summary>
        const PUDS_USBBUS2: TPUDSCANHandle = $52;
        /// <summary>
        /// PCAN-USB interface, channel 3
        /// </summary>
        const PUDS_USBBUS3: TPUDSCANHandle = $53;
        /// <summary>
        /// PCAN-USB interface, channel 4
        /// </summary>
        const PUDS_USBBUS4: TPUDSCANHandle = $54;
        /// <summary>
        /// PCAN-USB interface, channel 5
        /// </summary>
        const PUDS_USBBUS5: TPUDSCANHandle = $55;
        /// <summary>
        /// PCAN-USB interface, channel 6
        /// </summary>
        const PUDS_USBBUS6: TPUDSCANHandle = $56;
        /// <summary>
        /// PCAN-USB interface, channel 7
        /// </summary>
        const PUDS_USBBUS7: TPUDSCANHandle = $57;
        /// <summary>
        /// PCAN-USB interface, channel 8
        /// </summary>
        const PUDS_USBBUS8: TPUDSCANHandle = $58;
        /// <summary>
        /// PCAN-USB interface, channel 9
        /// </summary>
        const PUDS_USBBUS9: TPUDSCANHandle = $509;
        /// <summary>
        /// PCAN-USB interface, channel 10
        /// </summary>
        const PUDS_USBBUS10: TPUDSCANHandle = $50A;
        /// <summary>
        /// PCAN-USB interface, channel 11
        /// </summary>
        const PUDS_USBBUS11: TPUDSCANHandle = $50B;
        /// <summary>
        /// PCAN-USB interface, channel 12
        /// </summary>
        const PUDS_USBBUS12: TPUDSCANHandle = $50C;
        /// <summary>
        /// PCAN-USB interface, channel 13
        /// </summary>
        const PUDS_USBBUS13: TPUDSCANHandle = $50D;
        /// <summary>
        /// PCAN-USB interface, channel 14
        /// </summary>
        const PUDS_USBBUS14: TPUDSCANHandle = $50E;
        /// <summary>
        /// PCAN-USB interface, channel 15
        /// </summary>
        const PUDS_USBBUS15: TPUDSCANHandle = $50F;
        /// <summary>
        /// PCAN-USB interface, channel 16
        /// </summary>
        const PUDS_USBBUS16: TPUDSCANHandle = $510;

        /// <summary>
        /// PCAN-PC Card interface, channel 1
        /// </summary>
        const PUDS_PCCBUS1: TPUDSCANHandle = $61;
        /// <summary>
        /// PCAN-PC Card interface, channel 2
        /// </summary>
        const PUDS_PCCBUS2: TPUDSCANHandle = $62;

        /// <summary>
        /// PCAN-LAN interface, channel 1
        /// </summary>
        const PUDS_LANBUS1: TPUDSCANHandle = $801;
        /// <summary>
        /// PCAN-LAN interface, channel 2
        /// </summary>
        const PUDS_LANBUS2: TPUDSCANHandle = $802;
        /// <summary>
        /// PCAN-LAN interface, channel 3
        /// </summary>
        const PUDS_LANBUS3: TPUDSCANHandle = $803;
        /// <summary>
        /// PCAN-LAN interface, channel 4
        /// </summary>
        const PUDS_LANBUS4: TPUDSCANHandle = $804;
        /// <summary>
        /// PCAN-LAN interface, channel 5
        /// </summary>
        const PUDS_LANBUS5: TPUDSCANHandle = $805;
        /// <summary>
        /// PCAN-LAN interface, channel 6
        /// </summary>
        const PUDS_LANBUS6: TPUDSCANHandle = $806;
        /// <summary>
        /// PCAN-LAN interface, channel 7
        /// </summary>
        const PUDS_LANBUS7: TPUDSCANHandle = $807;
        /// <summary>
        /// PCAN-LAN interface, channel 8
        /// </summary>
        const PUDS_LANBUS8: TPUDSCANHandle = $808;
        /// <summary>
        /// PCAN-LAN interface, channel 9
        /// </summary>
        const PUDS_LANBUS9: TPUDSCANHandle = $809;
        /// <summary>
        /// PCAN-LAN interface, channel 10
        /// </summary>
        const PUDS_LANBUS10: TPUDSCANHandle = $80A;
        /// <summary>
        /// PCAN-LAN interface, channel 11
        /// </summary>
        const PUDS_LANBUS11: TPUDSCANHandle = $80B;
        /// <summary>
        /// PCAN-LAN interface, channel 12
        /// </summary>
        const PUDS_LANBUS12: TPUDSCANHandle = $80C;
        /// <summary>
        /// PCAN-LAN interface, channel 13
        /// </summary>
        const PUDS_LANBUS13: TPUDSCANHandle = $80D;
        /// <summary>
        /// PCAN-LAN interface, channel 14
        /// </summary>
        const PUDS_LANBUS14: TPUDSCANHandle = $80E;
        /// <summary>
        /// PCAN-LAN interface, channel 15
        /// </summary>
        const PUDS_LANBUS15: TPUDSCANHandle = $80F;
        /// <summary>
        /// PCAN-LAN interface, channel 16
        /// </summary>
        const PUDS_LANBUS16: TPUDSCANHandle = $810;


        /// <summary>
        /// No debug messages
        /// </summary>
        const PUDS_DEBUG_NONE: Byte = 0;
        /// <summary>
        /// Puts CAN debug messages to stdout
        /// </summary>
        const PUDS_DEBUG_CAN: Byte = 1;
        /// <summary>
        /// The Channel is illegal or not available
        /// </summary>
        const PUDS_CHANNEL_UNAVAILABLE: Byte = 0;
        /// <summary>
        /// The Channel is available
        /// </summary>
        const PUDS_CHANNEL_AVAILABLE: Byte = 1;
        /// <summary>
        /// The Channel is valid, and is being used
        /// </summary>
        const PUDS_CHANNEL_OCCUPIED: Byte = 2;

        /// <summary>
        /// Physical address for external test equipment
        /// </summary>
        const PUDS_SERVER_ADDR_TEST_EQUIPMENT: Byte = $F1;
        /// <summary>
        /// Functional request address for Legislated OBD system
        /// </summary>
        const PUDS_SERVER_ADDR_REQUEST_OBD_SYSTEM: Byte = $33;
        /// <summary>
        /// Flag stating that the address is defined as a ISO-15765-3 address
        /// </summary>
        const PUDS_SERVER_ADDR_FLAG_ENHANCED_ISO_15765_3: Word = $1000;
        /// <summary>
        /// Mask used for the ISO-15765-3 enhanced addresses
        /// </summary>
        const PUDS_SERVER_ADDR_MASK_ENHANCED_ISO_15765_3: Word = $07FF;
        /// <summary>
        /// Filter status : ignore (used to remove previously set filter)
        /// </summary>
        const PUDS_SERVER_FILTER_IGNORE: Word = $0000;
        /// <summary>
        /// Filter status : listen to (must be OR'ed with the 2 Byte functional address)
        /// </summary>
        const PUDS_SERVER_FILTER_LISTEN: Word = $8000;
        /// <summary>
        /// Default maximum timeout for UDS transmit confirmation
        /// </summary>
        const PUDS_TIMEOUT_REQUEST: LongWord = 10000;
        /// <summary>
        /// Default maximum timeout for UDS response reception
        /// </summary>
        const PUDS_TIMEOUT_RESPONSE: LongWord = 10000;
        /// <summary>
        /// Default server performance requirement (See ISO-15765-3 §6.3.2)
        /// </summary>
        const PUDS_P2CAN_DEFAULT_SERVER_MAX: Word = 50;
        /// <summary>
        /// Enhanced server performance requirement (See ISO-15765-3 §6.3.2)
        /// </summary>
        const PUDS_P2CAN_ENHANCED_SERVER_MAX: Word = 5000;
        /// <summary>
        /// Uses CAN frame data optimization
        /// </summary>
        const PUDS_CAN_DATA_PADDING_NONE:Word = $00;
        /// <summary>
        /// Uses CAN frame data padding (default, i.e. CAN DLC = 8)
        /// </summary>
        const PUDS_CAN_DATA_PADDING_ON:Word = $01;
        /// <summary>
        /// Default value used if CAN data padding is enabled
        /// </summary>
        const PUDS_CAN_DATA_PADDING_VALUE:Word = $55;


        /// <summary>
        /// Maximum data length  of UDS messages
        /// </summary>
        const PUDS_MAX_DATA: Word = 4095;
        /// <summary>
        /// Value (for member NO_POSITIVE_RESPONSE_MSG) stating to suppress positive response messages
        /// </summary>
        const PUDS_SUPPR_POS_RSP_MSG_INDICATION_BIT: Byte = $80;
        /// <summary>
        /// Default Value (for member NO_POSITIVE_RESPONSE_MSG) stating to keep positive response messages
        /// </summary>
        const PUDS_KEEP_POS_RSP_MSG_INDICATION_BIT: Byte = $00;
        /// <summary>
        /// Negative response code: Server wants more time
        /// </summary>
        const PUDS_NRC_EXTENDED_TIMING: Byte = $78;
        /// <summary>
        /// Positive response offset
        /// </summary>
        const PUDS_SI_POSITIVE_RESPONSE: Byte = $40;


        /// <summary>
        /// Initializes a PUDS-Client based on a PUDS-Channel
        /// </summary>
        /// <remarks>Only one UDS-Client can be initialized per CAN-Channel</remarks>
        /// <param name="CanChannel">The PCAN-Basic channel to be used as UDS client</param>
        /// <param name="Baudrate">The CAN Hardware speed</param>
        /// <param name="HwType">NON PLUG&PLAY: The type of hardware and operation mode</param>
        /// <param name="IOPort">NON PLUG&PLAY: The I/O address for the parallel port</param>
        /// <param name="Interrupt">NON PLUG&PLAY: Interrupt number of the parallel port</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function Initialize(
            CanChannel: TPUDSCANHandle;
            Baudrate: TPUDSBaudrate;
            HwType: TPUDSHWType;
            IOPort: LongWord;
            Interrupt: Word): TPUDSStatus; overload;
        /// <summary>
        /// Initializes a PUDS-Client based on a PUDS-Channel
        /// </summary>
        /// <remarks>Only one UDS-Client can be initialized per CAN-Channel</remarks>
        /// <param name="CanChannel">The PCAN-Basic channel to be used as UDS client</param>
        /// <param name="Baudrate">The CAN Hardware speed</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function Initialize(
            CanChannel: TPUDSCANHandle;
            Baudrate: TPUDSBaudrate): TPUDSStatus; overload;

        /// <summary>
        /// Uninitializes a PUDS-Client initialized before
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function Uninitialize(
            CanChannel: TPUDSCANHandle): TPUDSStatus;

        /// <summary>
        /// Resets the receive and transmit queues of a PUDS-Client
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function Reset(
            CanChannel: TPUDSCANHandle): TPUDSStatus;

        /// <summary>
        /// Gets information about the internal BUS status of a PUDS CAN-Channel.
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function GetStatus(
            CanChannel: TPUDSCANHandle): TPUDSStatus;

        /// <summary>
        /// Reads a PUDS message from the receive queue of a PUDS-Client
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        /// <param name="MessageBuffer">A TPUDSMsg structure buffer to store the PUDS message</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function Read(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg): TPUDSStatus;

        /// <summary>
        /// Transmits a PUDS message
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        /// <param name="MessageBuffer">A TPUDSMsg buffer with the message to be sent</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function Write(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg): TPUDSStatus;

        /// <summary>
        /// Retrieves a PUDS-Client value
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        /// <param name="Parameter">The Parameter: TPUDSParameter to get</param>
        /// <param name="StringBuffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function GetValue(
            CanChannel: TPUDSCANHandle;
            Parameter: TPUDSParameter;
            StringBuffer: PAnsiChar;
            BufferLength: LongWord): TPUDSStatus; overload;
        /// <summary>
        /// Retrieves a PUDS-Client value
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        /// <param name="Parameter">The Parameter: TPUDSParameter to get</param>
        /// <param name="NumericBuffer">Buffer for the parameter value</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function GetValue(
            CanChannel: TPUDSCANHandle;
            Parameter: TPUDSParameter;
            NumericBuffer: PLongWord;
            BufferLength: LongWord): TPUDSStatus; overload;

        /// <summary>
        /// Configures or sets a PUDS-Client value
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        /// <param name="Parameter">The Parameter: TPUDSParameter to set</param>
        /// <param name="NumericBuffer">Buffer with the value to be set</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SetValue(
            CanChannel: TPUDSCANHandle;
            Parameter: TPUDSParameter;
            NumericBuffer: PLongWord;
            BufferLength: LongWord): TPUDSStatus; overload;
        /// <summary>
        /// Configures or sets a PUDS-Client value
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel Handle representing a PUDS-Client</param>
        /// <param name="Parameter">The Parameter: TPUDSParameter to set</param>
        /// <param name="StringBuffer">Buffer with the value to be set</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SetValue(
            CanChannel: TPUDSCANHandle;
            Parameter: TPUDSParameter;
            StringBuffer: PAnsiChar;
            BufferLength: LongWord): TPUDSStatus; overload;


        /// <summary>
        /// Waits for a message (a response or a transmit confirmation) based on a UDS Message Request.
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        /// <param name="MessageBuffer">A TPUDSMsg structure buffer to store the PUDS response</param>
        /// <param name="MessageRequest">A sent TPUDSMsg message</param>
        /// <param name="IsWaitForTransmit">The message to wait for is a Transmit Confirmation or not</param>
        /// <param name="TimeInterval">Time interval to check for new message</param>
        /// <param name="Timeout">Maximum time to wait for the message</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function WaitForSingleMessage(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            var MessageRequest: TPUDSMsg;
            IsWaitForTransmit: Boolean;
            TimeInterval: LongWord;
            Timeout: LongWord): TPUDSStatus;

        /// <summary>
        /// Waits for multiple messages (multiple responses from a functional request for instance) based on a UDS Message Request.
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        /// <param name="Buffer">Buffer must be an array of 'MaxCount' entries (must have at least
        /// a size of iMaxCount * sizeof(TPUDSMsg) bytes</param>
        /// <param name="MaxCount">Size of the Buffer array (max. messages that can be received)</param>
        /// <param name="pCount">Buffer for the real number of messages read</param>
        /// <param name="MessageRequest">A sent TPUDSMsg message</param>
        /// <param name="TimeInterval">Time interval to check for new message</param>
        /// <param name="Timeout">Maximum time to wait for the message</param>
        /// <param name="TimeoutEnhanced">Maximum time to wait for the message in UDS Enhanced mode</param>
        /// <param name="WaitUntilTimeout">if <code>FALSE</code> the function is interrupted if pCount reaches MaxCount.</param>
		/// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success,
		///	PUDS_ERROR_OVERFLOW indicates success but Buffer was too small to hold all responses.</returns>
        class function WaitForMultipleMessage(
            CanChannel: TPUDSCANHandle;
            Buffer: PTPUDSMsg;
            MaxCount: LongWord;
            pCount: PLongWord;
            var MessageRequest: TPUDSMsg;
            TimeInterval: LongWord;
            Timeout: LongWord;
            TimeoutEnhanced: LongWord;
			WaitUntilTimeout: Boolean): TPUDSStatus;

        /// <summary>
        /// Handles the communication workflow for a UDS service expecting a single response.
        /// </summary>
        ///	<remark>
        ///	The function waits for a transmit confirmation then for a message response.
        ///	Even if the SuppressPositiveResponseMessage flag is set, the function will still wait
        /// for an eventual Negative Response.
        ///	</remark>
        /// <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        /// <param name="MessageBuffer">A TPUDSMsg structure buffer to store the PUDS response</param>
        /// <param name="MessageRequest">A sent TPUDSMsg message</param>
        /// <param name="MessageReqBuffer">A TPUDSMsg structure buffer to store the PUDS request confirmation
        ///	(if <code>NULL</code>, the result confirmation will be set in MessageRequest parameter)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function WaitForService(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            var MessageRequest: TPUDSMsg;
            MessageReqBuffer: PTPUDSMsg = nil): TPUDSStatus;

        /// <summary>
        /// Handles the communication workflow for a UDS service expecting multiple responses.
        /// </summary>
        ///	<remark>
        ///	The function waits for a transmit confirmation then for N message responses.
        ///	Even if the SuppressPositiveResponseMessage flag is set, the function will still wait
        /// for eventual Negative Responses.
        ///	</remark>
        /// <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        /// <param name="Buffer">Buffer must be an array of 'MaxCount' entries (must have at least
        /// a size of iMaxCount * sizeof(TPUDSMsg) Bytes</param>
        /// <param name="MaxCount">Size of the Buffer array (max. messages that can be received)</param>
        /// <param name="pCount">Buffer for the real number of messages read</param>
		/// <param name="WaitUntilTimeout">if <code>FALSE</code> the function is interrupted if pCount reaches MaxCount.</param>
        /// <param name="MessageRequest">A sent TPUDSMsg message</param>
        /// <param name="MessageReqBuffer">A TPUDSMsg structure buffer to store the PUDS request confirmation
        ///	(if <code>NULL</code>, the result confirmation will be set in MessageRequest parameter)</param>
		/// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success,
		///	PUDS_ERROR_OVERFLOW indicates success but Buffer was too small to hold all responses.</returns>
        class function WaitForServiceFunctional(
            CanChannel: TPUDSCANHandle;
            Buffer: PTPUDSMsg;
            MaxCount: LongWord;
            pCount: PLongWord;
            WaitUntilTimeout: Boolean;
            var MessageRequest: TPUDSMsg;
            MessageReqBuffer: PTPUDSMsg = nil): TPUDSStatus;

        /// <summary>
        /// Process a UDS response message to manage ISO-14229/15765 features (like session handling).
        /// </summary>
        /// <param name="CanChannel">A PUDS CAN-Channel representing a PUDS-Client</param>
        /// <param name="MessageBuffer">A TPUDSMsg structure buffer representing the PUDS Response Message</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function ProcessResponse(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg): TPUDSStatus;


        // ISO-15765-3:2004 §9.2.1 p.42 & ISO-14229-1:2006 §9.2 p.36

        /// <summary>
        /// The DiagnosticSessionControl service is used to enable different diagnostic sessions in the server.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="SessionType">Subfunction parameter: type of the session (see PUDS_SVC_PARAM_DSC_xxx)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcDiagnosticSessionControl(
	        CanChannel: TPUDSCANHandle;
	        var MessageBuffer: TPUDSMsg;
            SessionType: TPUDSSvcParamDSC): TPUDSStatus;


        // ISO-15765-3:2004 §9.2.2 p.42 && ISO-14229-1:2006 §9.3 p.42

        /// <summary>
        /// The ECUReset service is used by the client to request a server reset.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="ResetType">Subfunction parameter: type of Reset (see PUDS_SVC_PARAM_ER_xxx)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcECUReset(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            ResetType: TPUDSSvcParamER): TPUDSStatus;


        // ISO-15765-3:2004 §9.2.3 p.43 && ISO-14229-1:2006 §9.4 p.45

        /// <summary>
        /// SecurityAccessType : Request Seed and Send Key values
        /// </summary>
        const PUDS_SVC_PARAM_SA_RSD_1: Byte = $01;		// Request Seed
        const PUDS_SVC_PARAM_SA_RSD_3: Byte = $03;		// Request Seed
        const PUDS_SVC_PARAM_SA_RSD_5: Byte = $05;		// Request Seed
        const PUDS_SVC_PARAM_SA_RSD_MIN: Byte = $07;	// Request Seed (odd numbers)
        const PUDS_SVC_PARAM_SA_RSD_MAX: Byte = $5F;	// Request Seed (odd numbers)
        const PUDS_SVC_PARAM_SA_SK_2: Byte = $02;		// Send Key
        const PUDS_SVC_PARAM_SA_SK_4: Byte = $04;		// Send Key
        const PUDS_SVC_PARAM_SA_SK_6: Byte = $06;		// Send Key
        const PUDS_SVC_PARAM_SA_SK_MIN: Byte = $08;		// Send Key (even numbers)
        const PUDS_SVC_PARAM_SA_SK_MAX: Byte = $60;		// Send Key (even numbers)

        /// <summary>
        /// SecurityAccess service provides a means to access data and/or diagnostic services which have
        ///	restricted access for security, emissions or safety reasons.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="SecurityAccessType">Subfunction parameter: type of SecurityAccess (see PUDS_SVC_PARAM_SA_xxx)</param>
        /// <param name="Buffer">If Requesting Seed, buffer is the optional data to transmit to a server (like identification).
        ///	If Sending Key, data holds the value generated by the security algorithm corresponding to a specific "seed" value</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcSecurityAccess(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            SecurityAccessType: Byte;
            Buffer: PByte;
            BufferLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.2.4 p.43 && ISO-14229-1:2006 §9.5 p.52

        /// <summary>
        /// CommunicationType Flag: Application (01b)
        /// </summary>
        const PUDS_SVC_PARAM_CC_FLAG_APPL: Byte = $01;
        /// <summary>
        /// CommunicationType Flag: NetworkManagement (10b)
        /// </summary>
        const PUDS_SVC_PARAM_CC_FLAG_NWM: Byte = $02;
        /// <summary>
        /// CommunicationType Flag: Disable/Enable specified communicationType (see Flags APPL/NMW)
        /// </summary>
        const PUDS_SVC_PARAM_CC_FLAG_DESCTIRNCN: Byte = $00;
        // in the receiving node and all connected networks
        /// <summary>
        /// CommunicationType Flag: Disable/Enable network which request is received on
        /// </summary>
        const PUDS_SVC_PARAM_CC_FLAG_DENWRIRO: Byte = $F0;
        /// <summary>
        /// CommunicationType Flag: Disable/Enable specific network identified by network number (minimum value)
        /// </summary>
        const PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MIN: Byte = $10;
        /// <summary>
        /// CommunicationType Flag: Disable/Enable specific network identified by network number (maximum value)
        /// </summary>
        const PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MAX: Byte = $E0;
        /// <summary>
        /// CommunicationType Flag: Mask for DESNIBNN bits
        /// </summary>
        const PUDS_SVC_PARAM_CC_FLAG_DESNIBNN_MASK: Byte = $F0;

        /// <summary>
        ///	CommunicationControl service's purpose is to switch on/off the transmission
        ///	and/or the reception of certain messages of (a) server(s).
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="ControlType">Subfunction parameter: type of CommunicationControl (see PUDS_SVC_PARAM_CC_xxx)</param>
        /// <param name="CommunicationType">a bit-code value to reference the kind of communication to be controlled,
        ///	See PUDS_SVC_PARAM_CC_FLAG_xxx flags and ISO_14229-2006 §B.1 for bit-encoding</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcCommunicationControl(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            ControlType: TPUDSSvcParamCC;
	        CommunicationType: Byte): TPUDSStatus;


        // ISO-15765-3:2004 §9.2.5 p.43 && ISO-14229-1:2006 §9.6 p.55

        /// <summary>
        ///	TesterPresent service indicates to a server (or servers) that a client is still connected
        ///	to the vehicle and that certain diagnostic services and/or communications
        ///	that have been previously activated are to remain active.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="TesterPresentType">No Subfunction parameter by default (PUDS_SVC_PARAM_TP_ZSUBF)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcTesterPresent(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            TesterPresentType: TPUDSSvcParamTP = PUDS_SVC_PARAM_TP_ZSUBF): TPUDSStatus;


        // ISO-15765-3:2004 §9.2.6 p.44 && ISO-14229-1:2006 §9.8 p.63

        /// <summary>
        ///	SecuredDataTransmission service's purpose is to transmit data that is protected
        ///	against attacks from third parties, which could endanger data security.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="Buffer">buffer containing the data as processed by the Security Sub-Layer (See ISO-15764)</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcSecuredDataTransmission(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            Buffer: PByte;
            BufferLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.2.7 p.44 && ISO-14229-1:2006 §9.9 p.69

        /// <summary>
        ///	ControlDTCSetting service shall be used by a client to stop or resume the setting of
        ///	diagnostic trouble codes (DTCs) in the server(s).
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="DTCSettingType">Subfunction parameter (see PUDS_SVC_PARAM_CDTCS_xxx)</param>
        /// <param name="Buffer">This parameter record is user-optional and transmits data to a server when controlling the DTC setting.
        ///	It can contain a list of DTCs to be turned on or off.</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcControlDTCSetting(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            DTCSettingType: TPUDSSvcParamCDTCS;
            Buffer: PByte;
            BufferLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.2.8 p.44 && ISO-14229-1:2006 §9.10 p.73

        /// <summary>
        /// expected size of EventTypeRecord for ROE_STPROE
        /// </summary>
        const PUDS_SVC_PARAM_ROE_STPROE_LEN: Byte = 0;
        /// <summary>
        /// expected size of EventTypeRecord for ROE_ONDTCS
        /// </summary>
        const PUDS_SVC_PARAM_ROE_ONDTCS_LEN: Byte = 1;
        /// <summary>
        /// expected size of EventTypeRecord for ROE_OTI
        /// </summary>
        const PUDS_SVC_PARAM_ROE_OTI_LEN: Byte = 1;
        /// <summary>
        /// expected size of EventTypeRecord for ROE_OCODID
        /// </summary>
        const PUDS_SVC_PARAM_ROE_OCODID_LEN: Byte = 2;
        /// <summary>
        /// expected size of EventTypeRecord for ROE_RAE
        /// </summary>
        const PUDS_SVC_PARAM_ROE_RAE_LEN: Byte = 0;
        /// <summary>
        /// expected size of EventTypeRecord for ROE_STRTROE
        /// </summary>
        const PUDS_SVC_PARAM_ROE_STRTROE_LEN: Byte = 0;
        /// <summary>
        /// expected size of EventTypeRecord for ROE_CLRROE
        /// </summary>
        const PUDS_SVC_PARAM_ROE_CLRROE_LEN: Byte = 0;
        /// <summary>
        /// expected size of EventTypeRecord for ROE_OCOV
        /// </summary>
        const PUDS_SVC_PARAM_ROE_OCOV_LEN: Byte = 10;
        /// <summary>
        /// Infinite Time To Response (eventWindowTime parameter)
        /// </summary>
        const PUDS_SVC_PARAM_ROE_EWT_ITTR: Byte = $02;

        /// <summary>
        ///	The ResponseOnEvent service requests a server to
        ///	start or stop transmission of responses on a specified event.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="EventType">Subfunction parameter: event type (see PUDS_SVC_PARAM_ROE_xxx)</param>
        /// <param name="StoreEvent">Storage State (TRUE = Store Event, FALSE = Do Not Store Event)</param>
        /// <param name="EventWindowTime">Specify a window for the event logic to be active in the server (see PUDS_SVC_PARAM_ROE_EWT_ITTR)</param>
        /// <param name="EventTypeRecord">Additional parameters for the specified eventType</param>
        /// <param name="EventTypeRecordLength">Size in Bytes of the EventType Record (see PUDS_SVC_PARAM_ROE_xxx_LEN)</param>
        /// <param name="ServiceToRespondToRecord">Service parameters, with first Byte as service Id (see PUDS_SVC_PARAM_ROE_STRT_SI_xxx)</param>
        /// <param name="ServiceToRespondToRecordLength">Size in Bytes of the ServiceToRespondTo Record</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcResponseOnEvent(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            EventType: TPUDSSvcParamROE;
	        StoreEvent: Boolean;
	        EventWindowTime: Byte;
	        EventTypeRecord: PByte;
	        EventTypeRecordLength: Word;
	        ServiceToRespondToRecord: PByte;
	        ServiceToRespondToRecordLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.2.9 p.47 && ISO-14229-1:2006 §9.11 p.91

        /// <summary>
        ///	The LinkControl service is used to control the communication link baud rate
        ///	between the client and the server(s) for the exchange of diagnostic data.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="LinkControlType">Subfunction parameter: Link Control Type (see PUDS_SVC_PARAM_LC_xxx)</param>
        /// <param name="BaudrateIdentifier">defined baud rate identifier (see PUDS_SVC_PARAM_LC_BAUDRATE_xxx)</param>
        /// <param name="LinkBaudrate">used only with PUDS_SVC_PARAM_LC_VBTWSBR parameter:
        ///	a three-Byte value baud rate (baudrate High, Middle and Low Bytes).
        ///	</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success </returns>
        class function SvcLinkControl(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            LinkControlType: TPUDSSvcParamLC;
            BaudrateIdentifier: Byte;
            LinkBaudrate: LongWord): TPUDSStatus; overload;

        /// <summary>
        ///	The LinkControl service is used to control the communication link baud rate
        ///	between the client and the server(s) for the exchange of diagnostic data.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="LinkControlType">Subfunction parameter: Link Control Type (see PUDS_SVC_PARAM_LC_xxx)</param>
        /// <param name="BaudrateIdentifier">defined baud rate identifier (see PUDS_SVC_PARAM_LC_BAUDRATE_xxx)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success </returns>
        class function SvcLinkControl(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            LinkControlType: TPUDSSvcParamLC;
            BaudrateIdentifier: Byte): TPUDSStatus; overload;
        {
            return SvcLinkControl(CanChannel, var MessageBuffer, LinkControlType, BaudrateIdentifier, 0): TPUDSStatus;
        }


        // ISO-15765-3:2004 §9.3.1 p.47 && ISO-14229-1:2006 §10.2 p.97

        /// <summary>
        ///	The ReadDataByIdentifier service allows the client to request data record values
        ///	from the server identified by one or more dataIdentifiers.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="Buffer">buffer containing a list of two-Byte Data Identifiers (see PUDS_SVC_PARAM_DI_xxx)</param>
        /// <param name="BufferLength">Number of elements in the buffer (size in WORD of the buffer)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcReadDataByIdentifier(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            Buffer: PWord;
            BufferLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.3.2 p.47 && ISO-14229-1:2006 §10.3 p.102

        /// <summary>
        ///	The ReadMemoryByAddress service allows the client to request memory data from the server
        ///	via a provided starting address and to specify the size of memory to be read.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="MemoryAddress">starting address of server memory from which data is to be retrieved</param>
        /// <param name="MemoryAddressLength">Size in Bytes of the MemoryAddress buffer (max.: $F)</param>
        /// <param name="MemorySize">number of Bytes to be read starting at the address specified by memoryAddress</param>
        /// <param name="MemorySizeLength">Size in Bytes of the MemorySize buffer (max.: $F)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcReadMemoryByAddress(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            MemoryAddress: PByte;
            MemoryAddressLength: Byte;
            MemorySize: PByte;
            MemorySizeLength: Byte): TPUDSStatus;


        // ISO-15765-3:2004 §9.3.3 p.48 && ISO-14229-1:2006 §10.4 p.106

        /// <summary>
        ///	The ReadScalingDataByIdentifier service allows the client to request
        ///	scaling data record information from the server identified by a dataIdentifier.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="DataIdentifier">a two-Byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcReadScalingDataByIdentifier(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            DataIdentifier: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.3.4 p.48 && ISO-14229-1:2006 §10.5 p.112

        /// <summary>
        ///	The ReadDataByPeriodicIdentifier service allows the client to request the periodic transmission
        ///	of data record values from the server identified by one or more periodicDataIdentifiers.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="TransmissionMode">transmission rate code (see PUDS_SVC_PARAM_RDBPI_xxx)</param>
        /// <param name="Buffer">buffer containing a list of Periodic Data Identifiers</param>
        /// <param name="BufferLength">Number of elements in the buffer (size in WORD of the buffer)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcReadDataByPeriodicIdentifier(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
	        TransmissionMode: TPUDSSvcParamRDBPI;
	        Buffer: PByte;
	        BufferLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.3.5 p.54 && ISO-14229-1:2006 §10.6 p.123

        /// <summary>
        ///	The DynamicallyDefineDataIdentifier service allows the client to dynamically define
        ///	in a server a data identifier that can be read via the ReadDataByIdentifier service at a later time.
        ///	The Define By Identifier subfunction specifies that definition of the dynamic data
        ///	identifier shall occur via a data identifier reference.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="DynamicallyDefinedDataIdentifier">a two-Byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        /// <param name="SourceDataIdentifier">buffer containing the sources of information to be included into the dynamic data record</param>
        /// <param name="MemorySize">buffer containing the total numbers of Bytes from the source data record address</param>
        /// <param name="PositionInSourceDataRecord">buffer containing the starting Byte positions of the excerpt of the source data record</param>
        /// <param name="BuffersLength">Number of elements in the buffers (SourceDataIdentifier, MemoryAddress and PositionInSourceDataRecord)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcDynamicallyDefineDataIdentifierDBID(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
	        DynamicallyDefinedDataIdentifier: Word;
	        SourceDataIdentifier: PWord;
            MemorySize: PByte;
            PositionInSourceDataRecord: PByte;
	        BuffersLength: Word): TPUDSStatus;

        /// <summary>
        ///	The DynamicallyDefineDataIdentifier service allows the client to dynamically define
        ///	in a server a data identifier that can be read via the ReadDataByIdentifier service at a later time.
        ///	The Define By Memory Address subfunction specifies that definition of the dynamic data
        ///	identifier shall occur via an address reference.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="DynamicallyDefinedDataIdentifier">a two-Byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        /// <param name="MemoryAddressLength">Size in Bytes of the MemoryAddress items in the MemoryAddressBuffer buffer (max.: $F)</param>
        /// <param name="MemorySizeLength">Size in Bytes of the MemorySize items in the MemorySizeBuffer buffer (max.: $F)</param>
        /// <param name="MemoryAddressBuffer">buffer containing the MemoryAddress buffers,
        ///	must be an array of 'BuffersLength' entries which contains 'MemoryAddressLength' Bytes
        ///	(size is 'BuffersLength * MemoryAddressLength' Bytes)</param>
        /// <param name="MemorySizeBuffer">buffer containing the MemorySize buffers,
        ///	must be an array of 'BuffersLength' entries which contains 'MemorySizeLength' Bytes
        ///	(size is 'BuffersLength * MemorySizeLength' Bytes)</param>
        /// <param name="BuffersLength">Size in Bytes of the MemoryAddressBuffer and MemorySizeBuffer buffers</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcDynamicallyDefineDataIdentifierDBMA(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            DynamicallyDefinedDataIdentifier: Word;
            MemoryAddressLength: Byte;
            MemorySizeLength: Byte;
            MemoryAddressBuffer: PByte;
            MemorySizeBuffer: PByte;
            BuffersLength: Word): TPUDSStatus;

        /// <summary>
        ///	The Clear Dynamically Defined Data Identifier subfunction shall be used to clear
        ///	the specified dynamic data identifier.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="DynamicallyDefinedDataIdentifier">a two-Byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcDynamicallyDefineDataIdentifierCDDDI(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            DynamicallyDefinedDataIdentifier: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.3.6 p.54 && ISO-14229-1:2006 §10.7 p.143

        /// <summary>
        ///	The WriteDataByIdentifier service allows the client to write information into the server at an internal location
        ///	specified by the provided data identifier.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="DataIdentifier">a two-Byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        /// <param name="Buffer">buffer containing the data to write</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcWriteDataByIdentifier(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            DataIdentifier: Word;
            Buffer: PByte;
            BufferLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.3.7 p.54 && ISO-14229-1:2006 §10.8 p.146

        /// <summary>
        ///	The WriteMemoryByAddress service allows the client to write
        ///	information into the server at one or more contiguous memory locations.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="MemoryAddress">starting address of server memory to which data is to be written</param>
        /// <param name="MemoryAddressLength">Size in Bytes of the MemoryAddress buffer (max.: $F)</param>
        /// <param name="MemorySize">number of Bytes to be written starting at the address specified by memoryAddress</param>
        /// <param name="MemorySizeLength">Size in Bytes of the MemorySize buffer (max.: $F)</param>
        /// <param name="Buffer">buffer containing the data to write</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcWriteMemoryByAddress(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            MemoryAddress: PByte;
	        MemoryAddressLength: Byte;
            MemorySize: PByte;
	        MemorySizeLength: Byte;
            Buffer: PByte;
            BufferLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.4.2 p. && ISO-14229-1:2006 §11.2 p.152

        /// <summary>
        /// groupOfDTC : Emissions-related systems group of DTCs
        /// </summary>
        const PUDS_SVC_PARAM_CDI_ERS: LongWord = $000000;
        /// <summary>
        /// groupOfDTC : All Groups of DTCs
        /// </summary>
        const PUDS_SVC_PARAM_CDI_AGDTC: LongWord = $FFFFFF;

        /// <summary>
        ///	The ClearDiagnosticInformation service is used by the client to clear diagnostic information
        ///	in one server's or multiple servers' memory.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="groupOfDTC">a three-Byte value indicating the group of DTCs (e.g. powertrain, body, chassis)
        /// or the particular DTC to be cleared (see PUDS_SVC_PARAM_CDI_xxx)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcClearDiagnosticInformation(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            groupOfDTC: LongWord): TPUDSStatus;


        /// ISO-15765-3:2004 §9.4.1 p.54 && ISO-14229-1:2006 §11.3 p.154

        /// <summary>
        ///	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        /// Only reportNumberOfDTCByStatusMask, reportDTCByStatusMask, reportMirrorMemoryDTCByStatusMask,
        ///	reportNumberOfMirrorMemoryDTCByStatusMask, reportNumberOfEmissionsRelatedOBDDTCByStatusMask,
        ///	reportEmissionsRelatedOBDDTCByStatusMask Sub-functions are allowed.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="RDTCIType">Subfunction parameter: ReadDTCInformation type, use one of the following:
        ///	PUDS_SVC_PARAM_RDTCI_RNODTCBSM, PUDS_SVC_PARAM_RDTCI_RDTCBSM,
        ///	PUDS_SVC_PARAM_RDTCI_RMMDTCBSM, PUDS_SVC_PARAM_RDTCI_RNOMMDTCBSM,
        ///	PUDS_SVC_PARAM_RDTCI_RNOOBDDTCBSM, PUDS_SVC_PARAM_RDTCI_ROBDDTCBSM</param>
        /// <param name="DTCStatusMask">Contains eight DTC status bit.</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcReadDTCInformation(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            RDTCIType: TPUDSSvcParamRDTCI;
	        DTCStatusMask: Byte): TPUDSStatus;

        /// <summary>
        ///	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        ///	The sub-function reportDTCSnapshotRecordByDTCNumber (PUDS_SVC_PARAM_RDTCI_RDTCSSBDTC) is implicit.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="DTCMask">a unique identification number (three Byte value) for a specific diagnostic trouble code</param>
        /// <param name="DTCSnapshotRecordNumber">the number of the specific DTCSnapshot data records</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcReadDTCInformationRDTCSSBDTC(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            DTCMask:LongWord;
            DTCSnapshotRecordNumber: Byte): TPUDSStatus;

        /// <summary>
        ///	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        ///	The sub-function reportDTCSnapshotByRecordNumber (PUDS_SVC_PARAM_RDTCI_RDTCSSBRN) is implicit.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="DTCSnapshotRecordNumber">the number of the specific DTCSnapshot data records</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcReadDTCInformationRDTCSSBRN(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            DTCSnapshotRecordNumber: Byte): TPUDSStatus;

        /// <summary>
        ///	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        /// Only reportDTCExtendedDataRecordByDTCNumber and reportMirrorMemoryDTCExtendedDataRecordByDTCNumber Sub-functions are allowed.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="RDTCIType">Subfunction parameter: ReadDTCInformation type, use one of the following:
        ///	PUDS_SVC_PARAM_RDTCI_RDTCEDRBDN, PUDS_SVC_PARAM_RDTCI_RMMDEDRBDN</param>
        /// <param name="DTCMask">a unique identification number (three Byte value) for a specific diagnostic trouble code</param>
        /// <param name="DTCExtendedDataRecordNumber">the number of the specific DTCExtendedData record requested.</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcReadDTCInformationReportExtended(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            RDTCIType: TPUDSSvcParamRDTCI;
            DTCMask: LongWord;
            DTCExtendedDataRecordNumber: Byte): TPUDSStatus;

        /// <summary>
        ///	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        /// Only reportNumberOfDTCBySeverityMaskRecord and reportDTCSeverityInformation Sub-functions are allowed.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="RDTCIType">Subfunction parameter: ReadDTCInformation type, use one of the following:
        ///	PUDS_SVC_PARAM_RDTCI_RNODTCBSMR, PUDS_SVC_PARAM_RDTCI_RDTCBSMR</param>
        /// <param name="DTCSeverityMask">a mask of eight (8) DTC severity bits (see PUDS_SVC_PARAM_RDTCI_DTCSVM_xxx)</returns>
        /// <param name="DTCStatusMask">a mask of eight (8) DTC status bits</returns>
        class function SvcReadDTCInformationReportSeverity(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            RDTCIType: TPUDSSvcParamRDTCI;
            DTCSeverityMask: Byte;
            DTCStatusMask: Byte): TPUDSStatus;

        /// <summary>
        ///	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        ///	The sub-function reportSeverityInformationOfDTC (PUDS_SVC_PARAM_RDTCI_RSIODTC) is implicit.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="DTCMask">a unique identification number for a specific diagnostic trouble code</returns>
        class function SvcReadDTCInformationRSIODTC(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            DTCMask: LongWord): TPUDSStatus;

        /// <summary>
        ///	This service allows a client to read the status of server-resident Diagnostic Trouble Code (DTC) information.
        /// Only reportSupportedDTC, reportFirstTestFailedDTC, reportFirstConfirmedDTC, reportMostRecentTestFailedDTC,
        ///	reportMostRecentConfirmedDTC, reportDTCFaultDetectionCounter, reportDTCWithPermanentStatus,
        /// and reportDTCSnapshotIdentification Sub-functions are allowed.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="RDTCIType">Subfunction parameter: ReadDTCInformation type, use one of the following:
        ///	PUDS_SVC_PARAM_RDTCI_RFTFDTC, PUDS_SVC_PARAM_RDTCI_RFCDTC,
        ///	PUDS_SVC_PARAM_RDTCI_RMRTFDTC, PUDS_SVC_PARAM_RDTCI_RMRCDTC,
        ///	PUDS_SVC_PARAM_RDTCI_RSUPDTC, PUDS_SVC_PARAM_RDTCI_RDTCWPS,
        ///	PUDS_SVC_PARAM_RDTCI_RDTCSSI</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcReadDTCInformationNoParam(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            RDTCIType: TPUDSSvcParamRDTCI): TPUDSStatus;


        // ISO-15765-3:2004 §9.5.1 p.56 && ISO-14229-1:2006 §12.2 p.209

        /// <summary>
        ///	The InputOutputControlByIdentifier service is used by the client to substitute a value for an input signal,
        ///	internal server function and/or control an output (actuator) of an electronic system.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="DataIdentifier">a two-Byte Data Identifier (see PUDS_SVC_PARAM_DI_xxx)</param>
        /// <param name="ControlOptionRecord">First Byte can be used as either an InputOutputControlParameter
        ///	that describes how the server shall control its inputs or outputs (see PUDS_SVC_PARAM_IOCBI_xxx),
        ///	or as an additional controlState Byte</param>
        /// <param name="ControlOptionRecordLength">Size in Bytes of the ControlOptionRecord buffer</param>
        /// <param name="ControlEnableMaskRecord">The ControlEnableMask shall only be supported when
        ///	the inputOutputControlParameter is used and the dataIdentifier to be controlled consists
        ///	of more than one parameter (i.e. the dataIdentifier is bit-mapped or packeted by definition).
        ///	There shall be one bit in the ControlEnableMask corresponding to each individual parameter
        ///	defined within the dataIdentifier.</param>
        /// <param name="ControlEnableMaskRecordLength">Size in Bytes of the controlEnableMaskRecord buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcInputOutputControlByIdentifier(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            DataIdentifier: Word;
            ControlOptionRecord: PByte;
            ControlOptionRecordLength: Word;
            ControlEnableMaskRecord: PByte;
            ControlEnableMaskRecordLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.6.1 p.56 && ISO-14229-1:2006 §13.2 p.225

        /// <summary>
        ///	The RoutineControl service is used by the client to start/stop a routine,
        ///	and request routine results.
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message</param>
        /// <param name="RoutineControlType">Subfunction parameter: RoutineControl type (see PUDS_SVC_PARAM_RC_xxx)</param>
        /// <param name="RoutineIdentifier">Server Local Routine Identifier (see PUDS_SVC_PARAM_RC_RID_xxx)</param>
        /// <param name="Buffer">buffer containing the Routine Control Options (only with start and stop routine sub-functions)</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcRoutineControl(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            RoutineControlType: TPUDSSvcParamRC;
	        RoutineIdentifier: Word;
	        Buffer: PByte;
	        BufferLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.7.1 p.57 && ISO-14229-1:2006 §14.2 p.231

        /// <summary>
        ///	The requestDownload service is used by the client to initiate a data transfer
        ///	from the client to the server (download).
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="CompressionMethod">A nibble-value that specifies the "compressionMethod",
        ///	The value $0 specifies that no compressionMethod is used.</param>
        /// <param name="EncryptingMethod">A nibble-value that specifies the "encryptingMethod";
        ///	The value $0 specifies that no encryptingMethod is used.</param>
        /// <param name="MemoryAddress">starting address of server memory to which data is to be written</param>
        /// <param name="MemoryAddressLength">Size in Bytes of the MemoryAddress buffer (max.: $F)</param>
        /// <param name="MemorySize">used by the server to compare the uncompressed memory size with
        ///	the total amount of data transferred during the TransferData service</param>
        /// <param name="MemorySizeLength">Size in Bytes of the MemorySize buffer (max.: $F)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcRequestDownload(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
	        CompressionMethod: Byte;
	        EncryptingMethod: Byte;
	        MemoryAddress: PByte;
	        MemoryAddressLength: Byte;
	        MemorySize: PByte;
	        MemorySizeLength: Byte): TPUDSStatus;


        // ISO-15765-3:2004 §9.7.1 p.57 && ISO-14229-1:2006 §14.3 p.234

        /// <summary>
        ///	The requestUpload service is used by the client to initiate a data transfer
        ///	from the server to the client (upload).
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="CompressionMethod">A nibble-value that specifies the "compressionMethod",
        ///	The value $0 specifies that no compressionMethod is used.</param>
        /// <param name="EncryptingMethod">A nibble-value that specifies the "encryptingMethod",
        ///	The value $0 specifies that no encryptingMethod is used.</param>
        /// <param name="MemoryAddress">starting address of server memory from which data is to be retrieved</param>
        /// <param name="MemoryAddressLength">Size in Bytes of the MemoryAddress buffer (max.: $F)</param>
        /// <param name="MemorySize">used by the server to compare the uncompressed memory size with
        ///	the total amount of data transferred during the TransferData service</param>
        /// <param name="MemorySizeLength">Size in Bytes of the MemorySize buffer (max.: $F)</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcRequestUpload(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            CompressionMethod: Byte;
            EncryptingMethod: Byte;
            MemoryAddress: PByte;
            MemoryAddressLength: Byte;
            MemorySize: PByte;
            MemorySizeLength: Byte): TPUDSStatus;


        // ISO-15765-3:2004 §9.7.1 p.57 && ISO-14229-1:2006 §14.4 p.237

        /// <summary>
        ///	The TransferData service is used by the client to transfer data either from the client
        ///	to the server (download) or from the server to the client (upload).
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="BlockSequenceCounter">The blockSequenceCounter parameter value starts at 01 hex
        ///	with the first TransferData request that follows the RequestDownload (34 hex)
        ///	or RequestUpload (35 hex) service. Its value is incremented by 1 for each subsequent
        ///	TransferData request. At the value of FF hex, the blockSequenceCounter rolls over
        ///	and starts at 00 hex with the next TransferData request message.</param>
        /// <param name="Buffer">buffer containing the required transfer parameters</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcTransferData(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            BlockSequenceCounter: Byte;
            Buffer: PByte;
            BufferLength: Word): TPUDSStatus;


        // ISO-15765-3:2004 §9.7.1 p.57 && ISO-14229-1:2006 §14.5 p.242

        /// <summary>
        ///	The RequestTransferExit service is used by the client to terminate a data
        ///	transfer between client and server (upload or download).
        /// </summary>
        /// <param name="CanChannel">A PUDS Channel Handle representing a PUDS-Client</param>
        /// <param name="MessageBuffer">The PUDS message (NO_POSITIVE_RESPONSE_MSG is ignored)</param>
        /// <param name="Buffer">buffer containing the required transfer parameters</param>
        /// <param name="BufferLength">Size in Bytes of the buffer</param>
        /// <returns>A TPUDSStatus code. PUDS_ERROR_OK is returned on success</returns>
        class function SvcRequestTransferExit(
            CanChannel: TPUDSCANHandle;
            var MessageBuffer: TPUDSMsg;
            Buffer: PByte;
            BufferLength: Word): TPUDSStatus;

    end;


implementation
uses SysUtils;
const DLL_Name = 'PCAN-UDS.DLL';


function UDS_Initialize(CanChannel: TPUDSCANHandle; Baudrate: TPUDSBaudrate; HwType: TPUDSHWType; IOPort: LongWord; Interrupt: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_Uninitialize(CanChannel: TPUDSCANHandle): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_Reset(CanChannel: TPUDSCANHandle): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_GetStatus(CanChannel: TPUDSCANHandle): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_Read(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg):TPUDSStatus; stdcall;
external DLL_Name;
function UDS_Write(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_GetValue(CanChannel: TPUDSCANHandle; Parameter: TPUDSParameter; Buffer: Pointer; BufferLength: LongWord): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SetValue(CanChannel: TPUDSCANHandle; Parameter: TPUDSParameter; Buffer: Pointer; BufferLength: LongWord): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_WaitForSingleMessage(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; var MessageRequest: TPUDSMsg; IsWaitForTransmit: Boolean; TimeInterval: LongWord; Timeout: LongWord): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_WaitForMultipleMessage(CanChannel: TPUDSCANHandle; Buffer: PTPUDSMsg; MaxCount: LongWord; pCount: PLongWord; var MessageRequest: TPUDSMsg; TimeInterval: LongWord; Timeout: LongWord; TimeoutEnhanced: LongWord; WaitUntilTimeout: Boolean): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_WaitForService(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; var MessageRequest: TPUDSMsg; MessageReqBuffer: PTPUDSMsg = nil): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_WaitForServiceFunctional(CanChannel: TPUDSCANHandle; Buffer: PTPUDSMsg; MaxCount: LongWord; pCount: PLongWord; WaitUntilTimeout: Boolean; var MessageRequest: TPUDSMsg; MessageReqBuffer: PTPUDSMsg = nil): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_ProcessResponse(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg): TPUDSStatus; stdcall;
external DLL_Name;

function UDS_SvcDiagnosticSessionControl(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; SessionType: TPUDSSvcParamDSC): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcECUReset(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; ResetType: TPUDSSvcParamER): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcSecurityAccess(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; SecurityAccessType: Byte; Buffer: PByte; BufferLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcCommunicationControl(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; ControlType: TPUDSSvcParamCC; CommunicationType: Byte): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcTesterPresent(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; TesterPresentType: TPUDSSvcParamTP): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcSecuredDataTransmission(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; Buffer: PByte; BufferLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcControlDTCSetting(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DTCSettingType: TPUDSSvcParamCDTCS; Buffer: PByte; BufferLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcResponseOnEvent(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; EventType: TPUDSSvcParamROE; StoreEvent: Boolean; EventWindowTime: Byte; EventTypeRecord: PByte; EventTypeRecordLength: Word; ServiceToRespondToRecord: PByte; ServiceToRespondToRecordLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcLinkControl(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; LinkControlType: TPUDSSvcParamLC; BaudrateIdentifier: Byte; LinkBaudrate: LongWord): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadDataByIdentifier(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; Buffer: PWord; BufferLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadMemoryByAddress(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; MemoryAddress: PByte; MemoryAddressLength: Byte; MemorySize: PByte; MemorySizeLength: Byte): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadScalingDataByIdentifier(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DataIdentifier: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadDataByPeriodicIdentifier(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; TransmissionMode: TPUDSSvcParamRDBPI; Buffer: PByte; BufferLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcDynamicallyDefineDataIdentifierDBID(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DynamicallyDefinedDataIdentifier: Word; SourceDataIdentifier: PWord; MemorySize: PByte; PositionInSourceDataRecord: PByte; BuffersLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcDynamicallyDefineDataIdentifierDBMA(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DynamicallyDefinedDataIdentifier: Word; MemoryAddressLength: Byte; MemorySizeLength: Byte; MemoryAddressBuffer: PByte; MemorySizeBuffer: PByte; BuffersLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcDynamicallyDefineDataIdentifierCDDDI(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DynamicallyDefinedDataIdentifier: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcWriteDataByIdentifier(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DataIdentifier: Word; Buffer: PByte; BufferLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcWriteMemoryByAddress(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; MemoryAddress: PByte; MemoryAddressLength: Byte; MemorySize: PByte; MemorySizeLength: Byte; Buffer: PByte; BufferLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcClearDiagnosticInformation(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; groupOfDTC: LongWord): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadDTCInformation(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; RDTCIType: TPUDSSvcParamRDTCI; DTCStatusMask: Byte): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadDTCInformationRDTCSSBDTC(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DTCMask:LongWord; DTCSnapshotRecordNumber: Byte): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadDTCInformationRDTCSSBRN(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DTCSnapshotRecordNumber: Byte): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadDTCInformationReportExtended(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; RDTCIType: TPUDSSvcParamRDTCI; DTCMask: LongWord; DTCExtendedDataRecordNumber: Byte): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadDTCInformationReportSeverity(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; RDTCIType: TPUDSSvcParamRDTCI; DTCSeverityMask: Byte; DTCStatusMask: Byte): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadDTCInformationRSIODTC(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DTCMask: LongWord): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcReadDTCInformationNoParam(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; RDTCIType: TPUDSSvcParamRDTCI): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcInputOutputControlByIdentifier(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DataIdentifier: Word; ControlOptionRecord: PByte; ControlOptionRecordLength: Word; ControlEnableMaskRecord: PByte; ControlEnableMaskRecordLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcRoutineControl(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; RoutineControlType: TPUDSSvcParamRC; RoutineIdentifier: Word; Buffer: PByte; BufferLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcRequestDownload(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; CompressionMethod: Byte; EncryptingMethod: Byte; MemoryAddress: PByte; MemoryAddressLength: Byte; MemorySize: PByte; MemorySizeLength: Byte): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcRequestUpload(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; CompressionMethod: Byte; EncryptingMethod: Byte; MemoryAddress: PByte; MemoryAddressLength: Byte; MemorySize: PByte; MemorySizeLength: Byte): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcTransferData(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; BlockSequenceCounter: Byte; Buffer: PByte; BufferLength: Word): TPUDSStatus; stdcall;
external DLL_Name;
function UDS_SvcRequestTransferExit(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; Buffer: PByte; BufferLength: Word): TPUDSStatus; stdcall;
external DLL_Name;




class function TUdsApi.Initialize(CanChannel: TPUDSCANHandle; Baudrate: TPUDSBaudrate; HwType: TPUDSHWType; IOPort: LongWord; Interrupt: Word): TPUDSStatus;
begin
    Result:= UDS_Initialize(CanChannel, Baudrate, HwType, IOPort, Interrupt);
end;

class function TUdsApi.Initialize(CanChannel: TPUDSCANHandle; Baudrate: TPUDSBaudrate): TPUDSStatus;
begin
    Result:= UDS_Initialize(CanChannel, Baudrate, TPUDSHWType(0), 0, 0);
end;

class function TUdsApi.Uninitialize(CanChannel: TPUDSCANHandle): TPUDSStatus;
begin
    Result:= UDS_Uninitialize(CanChannel);
end;

class function TUdsApi.Reset(CanChannel: TPUDSCANHandle): TPUDSStatus;
begin
    Result:= UDS_Reset(CanChannel);
end;

class function TUdsApi.GetStatus(CanChannel: TPUDSCANHandle): TPUDSStatus;
begin
    Result:= UDS_GetStatus(CanChannel);
end;

class function TUdsApi.Read(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg):TPUDSStatus;
begin
    Result:= UDS_Read(CanChannel, MessageBuffer);
end;

class function TUdsApi.Write(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg): TPUDSStatus;
begin
    Result:= UDS_Write(CanChannel, MessageBuffer);
end;

class function TUdsApi.GetValue(CanChannel: TPUDSCANHandle; Parameter: TPUDSParameter; StringBuffer: PAnsiChar; BufferLength: LongWord): TPUDSStatus;
begin
    Result:= UDS_GetValue(CanChannel, Parameter, StringBuffer, BufferLength);
end;
class function TUdsApi.GetValue(CanChannel: TPUDSCANHandle; Parameter: TPUDSParameter; NumericBuffer: PLongWord; BufferLength: LongWord): TPUDSStatus;
begin
    Result:= UDS_GetValue(CanChannel, Parameter, NumericBuffer, BufferLength);
end;

class function TUdsApi.SetValue(CanChannel: TPUDSCANHandle; Parameter: TPUDSParameter; StringBuffer: PAnsiChar; BufferLength: LongWord): TPUDSStatus;
begin
    Result:= UDS_SetValue(CanChannel, Parameter, StringBuffer, BufferLength);
end;
class function TUdsApi.SetValue(CanChannel: TPUDSCANHandle; Parameter: TPUDSParameter; NumericBuffer: PLongWord; BufferLength: LongWord): TPUDSStatus;
begin
    Result:= UDS_SetValue(CanChannel, Parameter, NumericBuffer, BufferLength);
end;



class function TUdsApi.WaitForSingleMessage(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; var MessageRequest: TPUDSMsg; IsWaitForTransmit: Boolean; TimeInterval: LongWord; Timeout: LongWord): TPUDSStatus;
begin
    Result := UDS_WaitForSingleMessage(CanChannel, MessageBuffer, MessageRequest, IsWaitForTransmit, TimeInterval, Timeout);
end;

class function TUdsApi.WaitForMultipleMessage(CanChannel: TPUDSCANHandle; Buffer: PTPUDSMsg; MaxCount: LongWord; pCount: PLongWord; var MessageRequest: TPUDSMsg; TimeInterval: LongWord; Timeout: LongWord; TimeoutEnhanced: LongWord; WaitUntilTimeout: Boolean): TPUDSStatus;
begin
    Result := UDS_WaitForMultipleMessage(CanChannel, Buffer, MaxCount, pCount, MessageRequest, TimeInterval, Timeout, TimeoutEnhanced, WaitUntilTimeout);
end;

class function TUdsApi.WaitForService(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; var MessageRequest: TPUDSMsg; MessageReqBuffer: PTPUDSMsg = nil): TPUDSStatus;
begin
    Result := UDS_WaitForService(CanChannel, MessageBuffer, MessageRequest, MessageReqBuffer);
end;

class function TUdsApi.WaitForServiceFunctional(CanChannel: TPUDSCANHandle; Buffer: PTPUDSMsg; MaxCount: LongWord; pCount: PLongWord; WaitUntilTimeout: Boolean; var MessageRequest: TPUDSMsg; MessageReqBuffer: PTPUDSMsg = nil): TPUDSStatus;
begin
    Result := UDS_WaitForServiceFunctional(CanChannel, Buffer, MaxCount, pCount, WaitUntilTimeout, MessageRequest, MessageReqBuffer);
end;

class function TUdsApi.ProcessResponse(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg): TPUDSStatus;
begin
    Result := UDS_ProcessResponse(CanChannel, MessageBuffer);
end;


class function TUdsApi.SvcDiagnosticSessionControl(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; SessionType: TPUDSSvcParamDSC): TPUDSStatus;
begin
    Result := UDS_SvcDiagnosticSessionControl(CanChannel, MessageBuffer, SessionType);
end;
class function TUdsApi.SvcECUReset(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; ResetType: TPUDSSvcParamER): TPUDSStatus;
begin
    Result := UDS_SvcECUReset(CanChannel, MessageBuffer, ResetType);
end;
class function TUdsApi.SvcSecurityAccess(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; SecurityAccessType: Byte; Buffer: PByte; BufferLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcSecurityAccess(CanChannel, MessageBuffer, SecurityAccessType, Buffer, BufferLength);
end;
class function TUdsApi.SvcCommunicationControl(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; ControlType: TPUDSSvcParamCC; CommunicationType: Byte): TPUDSStatus;
begin
    Result := UDS_SvcCommunicationControl(CanChannel, MessageBuffer, ControlType, CommunicationType);
end;
class function TUdsApi.SvcTesterPresent(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; TesterPresentType: TPUDSSvcParamTP): TPUDSStatus;
begin
    Result := UDS_SvcTesterPresent(CanChannel, MessageBuffer, TesterPresentType);
end;
class function TUdsApi.SvcSecuredDataTransmission(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; Buffer: PByte; BufferLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcSecuredDataTransmission(CanChannel, MessageBuffer, Buffer, BufferLength);
end;
class function TUdsApi.SvcControlDTCSetting(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DTCSettingType: TPUDSSvcParamCDTCS; Buffer: PByte; BufferLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcControlDTCSetting(CanChannel, MessageBuffer, DTCSettingType, Buffer, BufferLength);
end;
class function TUdsApi.SvcResponseOnEvent(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; EventType: TPUDSSvcParamROE; StoreEvent: Boolean; EventWindowTime: Byte; EventTypeRecord: PByte; EventTypeRecordLength: Word; ServiceToRespondToRecord: PByte; ServiceToRespondToRecordLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcResponseOnEvent(CanChannel, MessageBuffer, EventType, StoreEvent, EventWindowTime, EventTypeRecord, EventTypeRecordLength, ServiceToRespondToRecord, ServiceToRespondToRecordLength);
end;
class function TUdsApi.SvcLinkControl(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; LinkControlType: TPUDSSvcParamLC; BaudrateIdentifier: Byte; LinkBaudrate: LongWord): TPUDSStatus;
begin
    Result := UDS_SvcLinkControl(CanChannel, MessageBuffer, LinkControlType, BaudrateIdentifier, LinkBaudrate);
end;
class function TUdsApi.SvcLinkControl(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; LinkControlType: TPUDSSvcParamLC; BaudrateIdentifier: Byte): TPUDSStatus;
begin
    Result := UDS_SvcLinkControl(CanChannel, MessageBuffer, LinkControlType, BaudrateIdentifier, 0);
end;
class function TUdsApi.SvcReadDataByIdentifier(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; Buffer: PWord; BufferLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcReadDataByIdentifier(CanChannel, MessageBuffer, Buffer, BufferLength);
end;
class function TUdsApi.SvcReadMemoryByAddress(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; MemoryAddress: PByte; MemoryAddressLength: Byte; MemorySize: PByte; MemorySizeLength: Byte): TPUDSStatus;
begin
    Result := UDS_SvcReadMemoryByAddress(CanChannel, MessageBuffer, MemoryAddress, MemoryAddressLength, MemorySize, MemorySizeLength);
end;
class function TUdsApi.SvcReadScalingDataByIdentifier(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DataIdentifier: Word): TPUDSStatus;
begin
    Result := UDS_SvcReadScalingDataByIdentifier(CanChannel, MessageBuffer, DataIdentifier);
end;
class function TUdsApi.SvcReadDataByPeriodicIdentifier(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; TransmissionMode: TPUDSSvcParamRDBPI; Buffer: PByte; BufferLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcReadDataByPeriodicIdentifier(CanChannel, MessageBuffer, TransmissionMode, Buffer, BufferLength);
end;
class function TUdsApi.SvcDynamicallyDefineDataIdentifierDBID(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DynamicallyDefinedDataIdentifier: Word; SourceDataIdentifier: PWord; MemorySize: PByte; PositionInSourceDataRecord: PByte; BuffersLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcDynamicallyDefineDataIdentifierDBID(CanChannel, MessageBuffer, DynamicallyDefinedDataIdentifier, SourceDataIdentifier, MemorySize, PositionInSourceDataRecord, BuffersLength);
end;
class function TUdsApi.SvcDynamicallyDefineDataIdentifierDBMA(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DynamicallyDefinedDataIdentifier: Word; MemoryAddressLength: Byte; MemorySizeLength: Byte; MemoryAddressBuffer: PByte; MemorySizeBuffer: PByte; BuffersLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcDynamicallyDefineDataIdentifierDBMA(CanChannel, MessageBuffer, DynamicallyDefinedDataIdentifier, MemoryAddressLength, MemorySizeLength, MemoryAddressBuffer, MemorySizeBuffer, BuffersLength);
end;
class function TUdsApi.SvcDynamicallyDefineDataIdentifierCDDDI(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DynamicallyDefinedDataIdentifier: Word): TPUDSStatus;
begin
    Result := UDS_SvcDynamicallyDefineDataIdentifierCDDDI(CanChannel, MessageBuffer, DynamicallyDefinedDataIdentifier);
end;
class function TUdsApi.SvcWriteDataByIdentifier(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DataIdentifier: Word; Buffer: PByte; BufferLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcWriteDataByIdentifier(CanChannel, MessageBuffer, DataIdentifier, Buffer, BufferLength);
end;
class function TUdsApi.SvcWriteMemoryByAddress(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; MemoryAddress: PByte; MemoryAddressLength: Byte; MemorySize: PByte; MemorySizeLength: Byte; Buffer: PByte; BufferLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcWriteMemoryByAddress(CanChannel, MessageBuffer, MemoryAddress, MemoryAddressLength, MemorySize, MemorySizeLength, Buffer, BufferLength);
end;
class function TUdsApi.SvcClearDiagnosticInformation(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; groupOfDTC: LongWord): TPUDSStatus;
begin
    Result := UDS_SvcClearDiagnosticInformation(CanChannel, MessageBuffer, groupOfDTC);
end;
class function TUdsApi.SvcReadDTCInformation(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; RDTCIType: TPUDSSvcParamRDTCI; DTCStatusMask: Byte): TPUDSStatus;
begin
    Result := UDS_SvcReadDTCInformation(CanChannel, MessageBuffer, RDTCIType, DTCStatusMask);
end;
class function TUdsApi.SvcReadDTCInformationRDTCSSBDTC(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DTCMask:LongWord; DTCSnapshotRecordNumber: Byte): TPUDSStatus;
begin
    Result := UDS_SvcReadDTCInformationRDTCSSBDTC(CanChannel, MessageBuffer, DTCMask, DTCSnapshotRecordNumber);
end;
class function TUdsApi.SvcReadDTCInformationRDTCSSBRN(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DTCSnapshotRecordNumber: Byte): TPUDSStatus;
begin
    Result := UDS_SvcReadDTCInformationRDTCSSBRN(CanChannel, MessageBuffer, DTCSnapshotRecordNumber);
end;
class function TUdsApi.SvcReadDTCInformationReportExtended(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; RDTCIType: TPUDSSvcParamRDTCI; DTCMask: LongWord; DTCExtendedDataRecordNumber: Byte): TPUDSStatus;
begin
    Result := UDS_SvcReadDTCInformationReportExtended(CanChannel, MessageBuffer, RDTCIType, DTCMask, DTCExtendedDataRecordNumber);
end;
class function TUdsApi.SvcReadDTCInformationReportSeverity(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; RDTCIType: TPUDSSvcParamRDTCI; DTCSeverityMask: Byte; DTCStatusMask: Byte): TPUDSStatus;
begin
    Result := UDS_SvcReadDTCInformationReportSeverity(CanChannel, MessageBuffer, RDTCIType, DTCSeverityMask, DTCStatusMask);
end;
class function TUdsApi.SvcReadDTCInformationRSIODTC(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DTCMask: LongWord): TPUDSStatus;
begin
    Result := UDS_SvcReadDTCInformationRSIODTC(CanChannel, MessageBuffer, DTCMask);
end;
class function TUdsApi.SvcReadDTCInformationNoParam(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; RDTCIType: TPUDSSvcParamRDTCI): TPUDSStatus;
begin
    Result := UDS_SvcReadDTCInformationNoParam(CanChannel, MessageBuffer, RDTCIType);
end;
class function TUdsApi.SvcInputOutputControlByIdentifier(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; DataIdentifier: Word; ControlOptionRecord: PByte; ControlOptionRecordLength: Word; ControlEnableMaskRecord: PByte; ControlEnableMaskRecordLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcInputOutputControlByIdentifier(CanChannel, MessageBuffer, DataIdentifier, ControlOptionRecord, ControlOptionRecordLength, ControlEnableMaskRecord, ControlEnableMaskRecordLength);
end;
class function TUdsApi.SvcRoutineControl(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; RoutineControlType: TPUDSSvcParamRC; RoutineIdentifier: Word; Buffer: PByte; BufferLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcRoutineControl(CanChannel, MessageBuffer, RoutineControlType, RoutineIdentifier, Buffer, BufferLength);
end;
class function TUdsApi.SvcRequestDownload(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; CompressionMethod: Byte; EncryptingMethod: Byte; MemoryAddress: PByte; MemoryAddressLength: Byte; MemorySize: PByte; MemorySizeLength: Byte): TPUDSStatus;
begin
    Result := UDS_SvcRequestDownload(CanChannel, MessageBuffer, CompressionMethod, EncryptingMethod, MemoryAddress, MemoryAddressLength, MemorySize, MemorySizeLength);
end;
class function TUdsApi.SvcRequestUpload(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; CompressionMethod: Byte; EncryptingMethod: Byte; MemoryAddress: PByte; MemoryAddressLength: Byte; MemorySize: PByte; MemorySizeLength: Byte): TPUDSStatus;
begin
    Result := UDS_SvcRequestUpload(CanChannel, MessageBuffer, CompressionMethod, EncryptingMethod, MemoryAddress, MemoryAddressLength, MemorySize, MemorySizeLength);
end;
class function TUdsApi.SvcTransferData(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; BlockSequenceCounter: Byte; Buffer: PByte; BufferLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcTransferData(CanChannel, MessageBuffer, BlockSequenceCounter, Buffer, BufferLength);
end;
class function TUdsApi.SvcRequestTransferExit(CanChannel: TPUDSCANHandle; var MessageBuffer: TPUDSMsg; Buffer: PByte; BufferLength: Word): TPUDSStatus;
begin
    Result := UDS_SvcRequestTransferExit(CanChannel, MessageBuffer, Buffer, BufferLength);
end;


end.

